import importlib
import os
import sys
import logging


import library.core.config_manager as config_manager
import library.core.references_solver as references_solver

importlib.reload(config_manager)

logger = logging.getLogger(__name__)
# logger.setLevel(logging.DEBUG)


class ProjectEnvironment():

    def __init__(self, project='', overrides=''):
        if not overrides:
            if 'SETTING_OVERRIDES' in os.environ and os.environ['SETTING_OVERRIDES']:
                overrides = os.environ['SETTING_OVERRIDES']

        self.project = None
        self.overrides = None
        self.update_solver(project, overrides=overrides)

    def update_solver(self, project, overrides=''):
        if project != self.project:
            self.project = project
            self.config_solver = config_manager.ConfigSolver(project=self.project)
            self._company_config = self.config_solver.company_config
            self._project_config = self.config_solver.get_config('project')

        if overrides != self.overrides:
            self.overrides = overrides
            if overrides:
                os.environ['SETTING_OVERRIDES'] = overrides
            self._project_tools_version = self.get_project_tools_version(overrides=self.overrides)



    def get_tool_versions(self, tool_name, tool_version, tool_type):
        tool_data = self.get_tool_config(tool_name, tool_version, tool_type)

        if tool_data.get('parent tool'):
            parent_data = self.get_tool_config(tool_data['parent tool'], tool_version, tool_type)
            if 'supported_versions' in parent_data:
                return parent_data['supported_versions']

    @property
    def project_tools(self):
        return self._project_tools_version

    @property
    def company_config(self):
        return self._company_config

    @property
    def project_config(self):
        return self._project_config

    def get_tool_config(self, tool_name, tool_version, tool_type):
        tool_version = str(tool_version)
        tool_config = self.config_solver.get_config(tool_name,
                                                    merge=False,
                                                    tool_type=tool_type,
                                                    version=tool_version,
                                                    module=tool_name)
        tool_path = self.config_solver.last_config_path
        return tool_config, tool_path

    def get_config_file(self, *args, **kargs):
        tool_config = self.config_solver.get_config(*args, **kargs)
        return tool_config


    def get_alt_versions(self, tool_name, tool_version):
        versions_dict = {'%s_version' % tool_name: str(tool_version),
                         '%s_mayor_version' % tool_name: str(tool_version).split('.')[0],
                         '%s_short_version' % tool_name: str(tool_version).split('v')[0],
                         '%s_compact_version' % tool_name: str(tool_version).replace('.', ''),
                         }
        return versions_dict

    def get_override_names(self):
        override_list = self.config_solver.get_config_files_of_type('test_settings')
        return override_list

    def get_project_tools(self, project=None, overrides=''):
        if project is not None:
            self.project = project

        self.update_solver(self.project, overrides=overrides)
        tools_config = {'tools_root': os.environ.get('TOOLS_ROOT', '')}
        versions_dict = {}
        for tool_type in self._project_tools_version:

            for tool_name in self.project_tools.get(tool_type):
                tool_version = self.project_tools[tool_type][tool_name]
                logger.info('Tool: %s %s' % (tool_name, tool_version))

                current_tool_type = tool_type

                tool_config, tool_path = self.get_tool_config(tool_name, tool_version, current_tool_type)
                logger.info(tool_path)

                if tool_config and tool_config.get('executable'):
                    for app in tool_config['executable']:
                        tool_config['executable'][app]['name'] = app
                        tool_config['executable'][app]['version'] = tool_version
                        tool_config['executable'][app]['tool_type'] = tool_type
                        tool_config['executable'][app]['path'] = os.path.dirname(tool_path).replace('\\', '/')
                        if app != tool_name:
                            tool_config['executable'][app]['parent tool'] = tool_name

                    tools_config.update(tool_config['executable'])

                tool_version_dict = self.get_alt_versions(tool_name, tool_version)
                versions_dict.update(tool_version_dict)
        tools_config = references_solver.ReferencesSolver.parse_dict(tools_config,
                                                                     aditional_values=versions_dict)

        return tools_config

    def get_project_tools_version(self, overrides=''):

        project_tools_versions = self.config_solver.get_config('project_tools')
        if overrides:
            override_data = self.config_solver.get_config('test_settings/%s' % overrides)
            self.config_solver.override_config(project_tools_versions, override_data)
        return project_tools_versions

    def get_environment(self, overrides=''):

        environment_variables = {}

        source_keys = ['ALLUSERSPROFILE',
                       'APPDATA',
                       'COMPANY_ROOT',
                       'TOOLS_ROOT',
                       'LOCAL_PIPELINE',
                       'COMMONPROGRAMFILES',
                       'COMMONPROGRAMFILES(X86)',
                       'COMMONPROGRAMW6432',
                       'COMPUTERNAME',
                       'COMSPEC',
                       'DEADLINE_PATH',
                       'DRIVERDATA',
                       'FPS_BROWSER_APP_PROFILE_STRING',
                       'FPS_BROWSER_USER_PROFILE_STRING',
                       'HOMEDRIVE',
                       'HOMEPATH',
                       'INTEL_DEV_REDIST',
                       'LOCALAPPDATA',
                       'LOGONSERVER',
                       'MIC_LD_LIBRARY_PATH',
                       'NINJARMMCLI',
                       'NUMBER_OF_PROCESSORS',
                       'ONEDRIVE',
                       'OS',
                       # 'PATH',
                       'PATHEXT',
                       'PROCESSOR_ARCHITECTURE',
                       'PROCESSOR_IDENTIFIER',
                       'PROCESSOR_LEVEL',
                       'PROCESSOR_REVISION',
                       'PROGRAMDATA',
                       'PROGRAMFILES',
                       'PROGRAMFILES(X86)',
                       'PROGRAMW6432',
                       'PROMPT',
                       'PSMODULEPATH',
                       'PUBLIC',
                       'PYTHONPATH',
                       'SESSIONNAME',
                       'SYSTEMDRIVE',
                       'SYSTEMROOT',
                       'TEMP',
                       'TMP',
                       'USERDOMAIN',
                       'USERDOMAIN_ROAMINGPROFILE',
                       'USERNAME',
                       'USERPROFILE',
                       'WINDIR']

        for key, value in os.environ.items():
            if key not in source_keys:
                continue
            value = value.split(';')
            environment_variables[key] = value

        path = r'C:\Program Files (x86)\Common Files\Intel\Shared Libraries\redist\intel64\compiler;C:\windows\system32;C:\windows;C:\windows\System32\Wbem;C:\windows\System32\WindowsPowerShell\v1.0\;C:\windows\System32\OpenSSH\;C:\Program Files (x86)\QuickTime\QTSystem\;C:\Users\Atlantis-user\AppData\Local\Microsoft\WindowsApps;'
        # environment_variables['PATH'] = path.split(';')

        if overrides:
            self.get_project_tools_version(overrides=overrides)
            environment_variables['SETTING_OVERRIDES'] = overrides

        for tool_type in self.project_tools:
            for tool_name in self.project_tools.get(tool_type):
                tool_version = self.project_tools[tool_type][tool_name]
                current_tool_type = tool_type
                config_data, tool_path = self.get_tool_config(tool_name, tool_version, current_tool_type)
                versions_dict = self.get_alt_versions(tool_name, tool_version)
                config_data = references_solver.ReferencesSolver.parse_dict(config_data, aditional_values=versions_dict)

                environment_variables['%s_VERSION' % tool_name.upper()] = [str(tool_version)]
                tool_folder = os.path.dirname(tool_path)

                for env_var, value in config_data.get('environment', {}).items():
                    if isinstance(value, str) and value.startswith('./'):
                        value = ('{}{}'.format(tool_folder, value[1:]))
                        value = value.replace('\\', '/')

                    environment_variables[env_var.upper()] = environment_variables.get(env_var.upper(), [])
                    if isinstance(value, list):
                        environment_variables[env_var.upper()] += value
                    else:
                        environment_variables[env_var.upper()].append(value)
        #
        environment_variables = references_solver.ReferencesSolver.parse_dict(environment_variables,
                                                                              start_delimiter='<', end_delimiter='>')
        environment_variables['PROJECT'] = [self.project]
        environment_variables['COMPANY_ROOT'] = [os.environ['COMPANY_ROOT']]

        return environment_variables

    def command_data(self, command_list, overrides=''):
        environment = self.get_environment(overrides)
        env = {}

        for env_var, values in environment.items():
            if isinstance(values, dict):
                continue
            if env_var in os.environ:
                if not isinstance(values, list):
                    values = values.split(';')

                added_values = os.environ[env_var]
                if added_values.find(';') > -1:
                    added_values = added_values.split(';')
                    values = values + added_values

            if len(values) == 1:
                env[env_var.upper()] = str(values[0])
                env[env_var.upper()] = env[env_var.upper()].replace('\\', '/')

            else:

                string = ';'.join([str(value) for value in values if value is not None and value != ''])
                env[env_var.upper()] = string.replace('\\', '/')

        return os.path.abspath(command_list), env

    def get_launch_data(self, tool_data, overrides={}):

        tool_name = tool_data['name']
        tool_version = tool_data['version']

        versions_dict = self.get_alt_versions(tool_data.get('parent tool', tool_name), tool_version)

        if 'location' in tool_data:
            tool_root = tool_data['location']
        else:
            tool_root = tool_data['path']

        tool_data['full_path'] = '%s/%s' % (tool_root, tool_data.get('executable', ''))

        command, environment = self.command_data(tool_data['full_path'], overrides)

        return command, environment


if __name__ == '__main__':
    import argparse
    from pprint import pprint
    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '-command', '--command', dest='command', help='the command to execute')
    parser.add_argument('-p', '-project', '--project', dest='project', help='the project')

    args = parser.parse_args()

    if len(sys.argv) == 1:
        cmd = ['D:/gaffer/bin/gaffer.cmd']
        cmd = ['D:/ProgramData/anaconda3/python.exe', 'D:/work/python/gridpandas/task_manager.py']
        project = 'tpt'
    else:
        cmd = [args.command]
        if cmd[0].endswith('.py'):
            cmd.insert(0, 'D:/ProgramData/anaconda3/python.exe')
        project = args.project

    project_environment = ProjectEnvironment(project=project, overrides='local_test')

    print('-' * 100)
    data = project_environment.get_project_tools(overrides='local_test')

    pprint(data['set_as_invisible_component'])
    #print(project_environment.get_launch_data(data['set_as_invisible_component'])[0])
    # data['maya']['name']= 'maya'
    # command, data = project_environment.get_launch_data(data['maya'])

