import importlib.resources as resources
import os

from PySide import QtCore, QtGui, QtWidgets

import launcher.config.apps.icons as apps_icons


class ProcessWidgets(QtWidgets.QFrame):
    close_signal = QtCore.Signal()

    def __init__(self,command, args=[], environment={}, detached=False, parent=None, project=None):
        super(ProcessWidgets, self).__init__(parent=parent)
        self.project = project

        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.command = command
        self.args = args
        self.environment = environment

        self.detached = detached
        self.process = None
        self.main_layout = QtWidgets.QHBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(20, 0, 0, 0)
        self.main_layout.setAlignment(QtCore.Qt.AlignCenter | QtCore.Qt.AlignVCenter)
        self.console_widget = QtWidgets.QTextEdit(self)
        self.main_layout.addWidget(self.console_widget)
        self.setLayout(self.main_layout)
        self.run()


    def run(self):
        
        if self.command.endswith('.py'):
            self.args.insert(0, self.command)
            tools_root = os.environ.get('TOOLS_ROOT','V:/company/tools')
            self.command = '%s/python/3.11/Scripts/python.exe' % tools_root

        if self.process is None:  # No process running.
            self.write_log("Executing process", msg_type='Process')
            self.process = QtCore.QProcess()  # Keep a reference to the QProcess (e.g. on self) while it's running.
            environ = QtCore.QProcessEnvironment()

            environ.insert('PROJECT', self.project)
            for key, value in self.environment.items():
                environ.insert(key, value)

            self.process.setProcessEnvironment(environ)

            self.process.readyReadStandardOutput.connect(self.handle_stdout)
            self.process.readyReadStandardError.connect(self.handle_stderr)
            self.process.stateChanged.connect(self.handle_state)
            self.process.finished.connect(self.process_finished)  # Clean up once complete.

            if self.detached:
                self.process.startDetached(self.command, self.args)
                self.write_log("Launched Detached", msg_type='Process')
            else:
                self.process.start(self.command, self.args)

    def write_log(self, message, msg_type='Info'):
        redColor = QtGui.QColor(239, 41, 41)
        greenColor = QtGui.QColor(78, 154, 6)
        blueColor = QtGui.QColor(114, 159, 207)
        orangeColor = QtGui.QColor(114, 159, 207)
        
        blackColor = QtGui.QColor(0, 0, 0)
        grayColor = QtGui.QColor(80, 80, 80)
        
        console = self.console_widget


        for line in message.split('\n'):
            if not line: 
                continue

            line = line.replace('\r', '')

            line_msg = line.split(':')[0]
            line_msg = line_msg.title()
            
            if line_msg == 'info':
                console.setTextColor(greenColor)
                console.append(line)
            
            elif line_msg == 'Warning':
                console.setTextColor(orangeColor)
                console.append(line)
            elif line_msg == 'Error':
                console.setTextColor(redColor)
                console.append(line)
            elif line_msg == 'Debug':
                console.setTextColor(grayColor)
                console.append(line)
                                
            elif msg_type =='Info':
                console.setTextColor(greenColor)
                console.append('Info: %s' % line)

            elif msg_type =='Process':
                console.setTextColor(blueColor)
                console.append('Process: %s' % line)

            elif msg_type =='Error':
                console.setTextColor(redColor)
                console.append('Error: %s' % line)

            else:
                console.setTextColor(blackColor)
                console.append(line)

    def handle_stderr(self):
        data = self.process.readAllStandardError()
        stderr = bytes(data).decode("utf8")
        self.write_log(stderr)

    def handle_stdout(self):
        data = self.process.readAllStandardOutput()
        stdout = bytes(data).decode("utf8")
        self.write_log(stdout)

    def handle_state(self, state):
        states = {
            QtCore.QProcess.NotRunning: 'Not running',
            QtCore.QProcess.Starting: 'Starting',
            QtCore.QProcess.Running: 'Running',
        }
        state_name = states[state]
        self.write_log(state_name, msg_type='Process')
        if state == QtCore.QProcess.NotRunning:
            self.process_finished()

    def process_finished(self):
        self.write_log("Process finished.", msg_type='Process')
        self.process = None

class WindowWidgets(QtWidgets.QFrame):
    close_signal =  QtCore.Signal()

    def __init__(self, parent=None, values=None):
        super(WindowWidgets, self).__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.main_layout = QtWidgets.QHBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(20, 0, 0, 0)
        self.main_layout.setAlignment(QtCore.Qt.AlignCenter | QtCore.Qt.AlignVCenter)

        self.setFixedHeight(40)
        self.label_widget = QtWidgets.QLabel('Tool Launcher')
        self.label_widget.setObjectName('Title')
        self.close_button = QtWidgets.QPushButton()
        self.close_button.clicked.connect(self.send_close)
        self.close_button.setObjectName('CloseButton')

        icon_object_path = resources.path(apps_icons,'cross.png')
        with icon_object_path as icon_path:
            pixmap = QtGui.QPixmap(str(icon_path))
        
        pixmap = pixmap.scaled(QtCore.QSize(32, 32))#), aspectRatioMode=QtCore.Qt.KeepAspectRatio)
       
        self.oldPos = self.pos()

        self.close_button.setIcon(pixmap)
        self.close_button.setFixedSize(32, 32)

        self.main_layout.addWidget(self.label_widget)
        self.main_layout.addStretch(1)
        self.main_layout.addWidget(self.close_button)
        
        self.setLayout(self.main_layout)
        self.installEventFilter(self)


    def send_close(self):
        self.close_signal.emit()

    def eventFilter(self, object, event):
        if object != self:
            return True
        if event.type() == QtCore.QEvent.HoverEnter or event.type() == QtCore.QEvent.Enter:
            self.setStyleSheet("WindowWidgets{border: 1px solid rgb(255, 166, 0);}")
            return True 
        elif event.type() == QtCore.QEvent.MouseButtonPress:
            self.oldPos = event.globalPos()
            return True

        elif event.type() == QtCore.QEvent.MouseMove:
            delta = QtCore.QPoint (self.mapToParent(event.globalPos()) - self.oldPos)
            self.window().move(self.window().x() + delta.x(), self.window().y() + delta.y())
            self.oldPos = self.mapToParent(event.globalPos())
            return True

        elif event.type() == QtCore.QEvent.HoverLeave or event.type() == QtCore.QEvent.Leave:
            self.setStyleSheet("WindowWidgets{border: 1px solid rgb(51, 68, 85);}")           
            return True
        return False



class LabeledWidget(QtWidgets.QFrame):
    arrange = 'vertical'
    font_style_sheet = "color: rgb(210,210,210);"
    def __init__(self, value, label, parent=None, values=None):
        super(LabeledWidget, self).__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.label = label
        self.value = value
        if values is None:
            self.values = []
        else:
            self.values = values

        if self.arrange == 'vertical':
            self.layout = QtWidgets.QVBoxLayout()
        else:
            self.layout = QtWidgets.QHBoxLayout()

        self.layout.setSpacing(2)
        self.layout.setContentsMargins(2, 2, 2, 2)

        self.label_widget = QtWidgets.QLabel('%s :' % self.label.title())
        self.label_widget.setFixedWidth(110)
        self.label_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)
        
        self.setLayout(self.layout)

        self.layout.addWidget(self.label_widget)

class LabeledCombo(LabeledWidget):
    currentIndexChanged =  QtCore.Signal(int)

    arrange = 'horizontal'
    def __init__(self, value, label, parent=None, values=None):
        super(LabeledCombo, self).__init__(value, label, parent=parent, values=values)
        self.label_widget.setObjectName('LabelCombo')

        self.add_value_widget()
        
        #self.value_widget.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)
        self.label_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)

        if values is None:
            values = [value]
    
        self.setValues(values)
        self.setValue(value)

    def add_value_widget(self):
        font = QtGui.QFont()
        font.setBold(True)

        self.value_widget = QtWidgets.QComboBox()
        self.value_widget.setObjectName('LabeledCombo')
        self.value_widget.setFixedWidth(150)
        self.value_widget.setFont(font)
        self.layout.addWidget(self.value_widget)
        self.value_widget.currentIndexChanged.connect(self.index_changed)
        self.layout.addStretch(1)

    def index_changed(self, index):
        self.currentIndexChanged.emit(index)

    def setValue(self, value):
        index = self.value_widget.findText(value, QtCore.Qt.MatchFixedString)
        if index >= 0:
            self.value_widget.setCurrentIndex(index)
        self.value = value

    def setValues(self, values):
        self.values = list(sorted(values))
        self.value_widget.clear()
        self.value_widget.addItems(values)

    def getValue(self):
        value = (self.value_widget.currentText())
        return value
    

class OptionsFrame(QtWidgets.QFrame):
    changed_project = QtCore.Signal(str)
    changed_department = QtCore.Signal(str)
    changed_overrides = QtCore.Signal(str)

    def __init__(self, project, department, projects, departments, overrides, parent=None):
        super(OptionsFrame, self).__init__(parent=parent)
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(5, 5, 5, 5)
        self.setFixedWidth(240)
        self.project = project
        self.projects = projects
        self.department = department
        self.departments = departments
        self.overrides = overrides
        self.overrides.insert(0, '-----')
        self.project_chooser = LabeledCombo(self.project, 'Projects',values=self.projects, parent=self)
        self.department_chooser = LabeledCombo(self.department, 'Department', values=self.departments, parent=self)
        self.overrides_chooser = LabeledCombo('-----', 'Test Settings', values=overrides, parent=self)

        self.project_chooser.currentIndexChanged.connect(self.send_changed_project)
        self.department_chooser.currentIndexChanged.connect(self.send_changed_department)
        self.overrides_chooser.currentIndexChanged.connect(self.send_changed_overrides)

        self.main_layout.addWidget(self.project_chooser)
        self.main_layout.addWidget(self.department_chooser)
        self.main_layout.addWidget(self.overrides_chooser)

        self.main_layout.addStretch(1)
        self.setLayout(self.main_layout)

    def send_changed_project(self, value):
        project = self.project_chooser.value_widget.itemText(value)
        self.changed_project.emit(project)

    def send_changed_overrides(self, value):
        overrides = self.overrides_chooser.value_widget.itemText(value)
        self.changed_overrides.emit(overrides)

    
    def send_changed_department(self, value):
        department = self.department_chooser.value_widget.itemText(value)
        self.changed_department.emit(department)
        

class ToolCard(QtWidgets.QWidget):
    tool_launched = QtCore.Signal(str)
    def __init__(self, label, tools_data, parent=None):
        super(ToolCard, self).__init__(parent=parent)
        self.tool_data = tools_data

        self.name = label
        self.tool_data['name'] = label.lower()
        tool_tip = self.tool_data.get('tool tip')
        if tool_tip:
            self.setToolTip(tool_tip)

        self.layout = QtWidgets.QHBoxLayout()
        self.layout.setSpacing(5)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.thumbnail = ThumbnailFrame(label, tools_data)

        self.layout.addWidget(self.thumbnail)
        self.setLayout(self.layout)
        self.thumbnail.clicked.connect(self.clicked)

    def clicked(self):
        self.tool_launched.emit(self.name)
    
    def getVersion(self):
        return self.thumbnail.version

class SelectVersionWidget(QtWidgets.QDialog):

    def __init__(self, label, version, versions=[], parent=None):
        super(SelectVersionWidget, self).__init__(parent)
        # Create widgets

        self.setWindowTitle('Override Version')
        self.version_widget = LabeledCombo(version, label, values=versions)
        self.buttons = QtWidgets.QDialogButtonBox(self)
        self.buttons.setOrientation(QtCore.Qt.Horizontal)
        self.buttons.addButton("Cancel", QtWidgets.QDialogButtonBox.RejectRole)
        self.buttons.addButton("Override", QtWidgets.QDialogButtonBox.AcceptRole)

        layout = QtWidgets.QVBoxLayout()

        layout.addWidget(self.version_widget)
        layout.addWidget(self.buttons)
        self.setLayout(layout)

        self.buttons.accepted.connect(self.accept)
        self.buttons.rejected.connect(self.reject)


class ThumbnailFrame(QtWidgets.QFrame):
    clicked = QtCore.Signal()

    def __init__(self, name, tool_data, parent=None):
        super(ThumbnailFrame, self).__init__(parent=parent)

        self.tool_data = tool_data
        self.icon_path = '%s/%s' % (tool_data['path'], tool_data['icon'])
        self.tool_name = name
        self.tool_tip = tool_data.get('tool tip', '')
        self.version = str(tool_data.get('version', 'NaN'))
        self.default_version = self.version
        self.card = parent
        self.icon_size = 40

        self.layout = QtWidgets.QHBoxLayout()
        self.layout.setSpacing(5)
        self.layout.setContentsMargins(10, 4, 4, 4)
        self.icon_Label = QtWidgets.QLabel()
        self.icon_Label.setObjectName('Icon')    
        self.icon_Label.setAlignment(QtCore.Qt.AlignCenter | QtCore.Qt.AlignVCenter)

        pixmap = QtGui.QPixmap(self.icon_path)
        pixmap = pixmap.scaled(QtCore.QSize(self.icon_size-6, self.icon_size-6))#, aspectRatioMode=QtCore.Qt.KeepAspectRatio)
       
        self.icon_Label.setPixmap(pixmap)
        self.icon_Label.setFixedSize(self.icon_size, self.icon_size)

        self.data_layout = QtWidgets.QHBoxLayout()

        self.version_widget = QtWidgets.QLabel('v%s' % self.version)
        self.version_widget.setObjectName('Version')
        self.version_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignTop)


        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(12)
          
        self.label_widget = QtWidgets.QLabel(self.tool_name)
        self.label_widget.setFont(font)
        self.label_widget.setObjectName('Label')

        self.layout.addWidget(self.icon_Label)     

        self.data_layout.addWidget(self.label_widget)
        self.data_layout.addWidget(self.version_widget)

        self.layout.addLayout(self.data_layout)


        self.setLayout(self.layout)
        self.installEventFilter(self)
    
    def set_version(self, new_version):
        if new_version:
            self.version = str(new_version)
            
        else:
            self.version = self.default_version
        
        self.tool_data['version'] = self.version
        if self.version == 'local':
            self.version_widget.setText(self.version)
            self.version_widget.setStyleSheet('QLabel{color: rgb(255, 166, 0);}')

        else:
            if self.version == self.default_version:
                self.version_widget.setStyleSheet('QLabel{color: rgb(210, 210, 250);}')
            else:
                self.version_widget.setStyleSheet('QLabel{color: rgb(21, 210, 25);}')
            self.version_widget.setText('v%s' % self.version)

        
    def eventFilter(self, object, event):

        if object != self:
            return False
        if event.type() == QtCore.QEvent.MouseButtonPress:
            if event.button() == QtCore.Qt.RightButton:
                 pass
            elif event.button() == QtCore.Qt.LeftButton:
                self.clicked.emit()
            return True

        elif event.type() == QtCore.QEvent.MouseMove:
            return True

        elif event.type() == QtCore.QEvent.HoverEnter:
            self.icon_Label.setStyleSheet("border: 1px solid rgb(50, 150, 250);")
            self.setStyleSheet("ThumbnailFrame{border: 1px solid rgb(50, 150, 250);}")
            self.label_widget.setStyleSheet("color: rgb(255, 166, 0);")
        
            return True
        
        elif event.type() == QtCore.QEvent.HoverLeave:
            self.icon_Label.setStyleSheet("border: 1px solid rgb(51, 68, 85);")
            self.setStyleSheet("ThumbnailFrame{border: 1px solid rgb(51, 68, 85);}")
            self.label_widget.setStyleSheet("color: rgb(50, 150, 250);")

            return True

        elif event.type() == QtCore.QEvent.ContextMenu:
            print('context menu')
            return
            menu = QtWidgets.QMenu()
            set_version = QtWidgets.QAction('Set Version')
            if 'parent tool' not in self.tool_data:
                menu.addAction(set_version)
                
            menu_click = menu.exec_(event.globalPos())
            if menu_click == set_version:
                tools_table = self.parent().parent()
                
                versions = tools_table.project_environment.get_tool_versions(self.tool_name, self.version, self.tool_data['tool_type'])
                form = SelectVersionWidget(self.tool_name, self.version, versions=versions, parent=self.window())
                output = form.exec_()
                if output:
                    new_version = str(form.version_widget.getValue())
                    self.set_version(new_version)
                

            return True
        return False
    

    def launch_task(self):
        print('clicked')