import importlib
import os
import sys
from pprint import pprint

from PySide import QtCore, QtWidgets

import library.core.config_manager as config_manager
import library.core.references_solver as references_solver

import launcher.config as launcher_resources
import launcher.lib.set_environment as set_environment
import launcher.lib.widgets as widgets
import launcher.lib.update_customizer as update_customizer

importlib.reload(set_environment)

class ToolsTable(QtWidgets.QWidget):

    def __init__(self, project='tpt', parent=None):
        super(ToolsTable, self).__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.project = project

        self.column_count = 3
        self.main_layout = QtWidgets.QGridLayout()

        self.setLayout(self.main_layout)
        self.current_department = 'td'
        self.tools_widgets = {}
        self.row_index = 0
        self.column_index = 0

        self.update_project(self.project)

    def update_project(self, project):
        self.project = project
        self.config_solver = config_manager.ConfigSolver(project=self.project)
        self.project_environment = set_environment.ProjectEnvironment(project=self.project)
        self.clean_tools()
        self.read_config(self.current_department)

    def clean_tools(self):
        for index_y in range(self.row_index + 1):
            for index_x in range(self.column_count + 1):
                to_delete = self.main_layout.itemAtPosition(index_y, index_x)
                if to_delete:
                    to_delete.widget().deleteLater()

        self.tools_widgets = {}
        self.row_index = 0
        self.column_index = 0

    def read_config(self, department='td', update=True, overrides=None):
        self.current_department = department
        self.current_overrides = overrides

        if overrides == '-----':
            overrides = None
        self.project_tools = self.project_environment.get_project_tools(project=self.project, overrides=overrides)

        launcher_version = self.project_environment.project_tools['libraries']['launcher']

        launcher_tools_settings = self.project_environment.config_solver.get_config('launcher_layout',
                                                                                    module='launcher',
                                                                                    tool_type='default',
                                                                                    version=launcher_version)

        for tool_name in sorted(self.project_tools.keys()):
            tool_data = self.project_tools[tool_name]
            if not launcher_tools_settings.get('apps'):
                continue

            if tool_name not in launcher_tools_settings['apps']['all'] and tool_name not in launcher_tools_settings[
                'apps'].get(department, {}):
                continue

            self.add_tool(tool_name, tool_data)

    def add_tool(self, tool_name, tools_data, tooltip=''):

        tool_widget = widgets.ToolCard(tool_name.title(), tools_data, parent=self)
        tool_widget.setToolTip(tooltip)
        tool_widget.tool_launched.connect(self.launch_tool)
        self.tools_widgets[tool_name] = tool_widget
        self.main_layout.addWidget(tool_widget, self.row_index, self.column_index)
        self.column_index += 1

        if self.column_index == self.column_count:
            self.column_index = 0
            self.row_index += 1

    def get_tool_data(self, tool_name):
        tool_widget = self.tools_widgets.get(tool_name.lower())
        if tool_widget:
            tool_data = tool_widget.tool_data
        else:
            tool_data = self.project_tools[tool_name]

        tool_data['name'] = tool_name

        return tool_data

    def get_launch_data(self, tool_data):
        tool_name = tool_data['name']
        tool_version = tool_data['version']
        tool_data['company_root'] = os.environ.get('COMPANY_ROOT', r'\\company\company')
        tool_data['tools_root'] = os.environ.get('TOOLS_ROOT', 'C:/pipeline/tools')
        if tool_data.get('parent tool'):
            parent_tool = tool_data['parent tool']
            tool_data[parent_tool] = tool_version
            tool_data['%s_version' % parent_tool] = str(tool_version)
            tool_data['%s_main' % parent_tool] = str(tool_version).split('.')[0]
            tool_data['%s_short' % parent_tool] = str(tool_version).split('v')[0]

        else:
            tool_data[tool_name] = tool_version
            tool_data['%s_version' % tool_name] = str(tool_version)
            tool_data['%s_main' % tool_name] = str(tool_version).split('.')[0]
            tool_data['%s_short' % tool_name] = str(tool_version).split('v')[0]

        versions = {}
        for tool_widget in self.tools_widgets:
            versions[tool_widget] = self.tools_widgets[tool_widget].getVersion()

        # tool_version = self.tools_widgets[tool_name].getVersion()

        if 'location' in tool_data:
            tool_root = tool_data['location']
        else:
            tool_root = tool_data['path']

        tool_data['full_path'] = '%s/%s' % (tool_root, tool_data['executable'])
        tools_data_solved = references_solver.ReferencesSolver.parse_dict(tool_data)
        command, environment = self.project_environment.command_data(tools_data_solved['full_path'], self.project,
                                                                     overrides=versions)
        return command, environment

    def launch_tool(self, tools_name):

        tool_data = self.get_tool_data(tools_name)

        command, environment = self.get_launch_data(tool_data)

        if command.endswith('.py') and not tool_data.get('standalone'):
            tool_data['standalone'] = 'python'
        args = []
        for arg, value in tool_data.get('args', {}).items():
            args.append('--%s' % str(arg.replace('_', '-')))
            if value == '<project>':
                value = self.project
            args.append(str(value))

        if tool_data.get('standalone'):
            standalone = tool_data.get('standalone')
            standalone_data = self.get_tool_data(standalone)
            standalone_command, standalone_environment = self.get_launch_data(standalone_data)
            args.insert(0, command.replace('\\', '/'))

            command = standalone_command
            environment = standalone_environment

        print('Running:', command, os.path.exists(command))
        print('args: % s' % ' '.join(args))

        if not os.path.exists(command):
            print('can\'t find the executable: %s' % command)
            return -1

        # environment['PYTHONPATH'] = ';'.join(sys.path)
        if tool_data.get('detached', False):
            import subprocess
            args.insert(0, command)
            subprocess.Popen(args, env=environment)
        else:
            new_index = self.parent().log_consoles.addTab(widgets.ProcessWidgets(command,
                                                                                 args=args,
                                                                                 environment=environment,
                                                                                 project=self.project),
                                                          tools_name)
            self.parent().log_consoles.setCurrentIndex(new_index)
            self.parent().log_consoles.setHidden(False)
            self.window().setFixedHeight(800)


class ToolsLauncherWidget(QtWidgets.QWidget):
    widget_mappers = {}
    settings_changed = QtCore.Signal()
    def __init__(self, project, parent=None):
        super(ToolsLauncherWidget, self).__init__(parent=parent)
        self.stored_widget_map = {}
        app_name = 'ToolsLauncher'
        self.settings = QtCore.QSettings(app_name, "settings")
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        update_customizer.check_sitecustomize_files()

        self.project = project
        self.project_environment = set_environment.ProjectEnvironment(project=self.project)

        launcher_version = self.project_environment.project_tools['libraries']['launcher']

        launcher_tools_settings = self.project_environment.config_solver.get_config('launcher_layout',
                                                                                    module='launcher',
                                                                                    version=launcher_version)

        self.layout = QtWidgets.QVBoxLayout()
        self.layout.setSpacing(10)
        self.layout.setContentsMargins(20, 10, 20, 0)
        self.window_widget = widgets.WindowWidgets()
        departments = launcher_tools_settings['apps'].keys()
        projects = []
        for project_name in self.project_environment.company_config['active_projects']:
            project_folder = self.project_environment.company_config['projects_roots'][project_name]
            full_name = os.path.join(project_folder, 'config', 'project.yaml')
            if os.path.exists(full_name):
                projects.append(project_name)

        overrides = self.project_environment.config_solver.get_config_files_of_type('test_settings')
        self.window_widget.close_signal.connect(self.close_app)

        self.content_layout = QtWidgets.QHBoxLayout()
        self.options_chooser = widgets.OptionsFrame(self.project, 'td', projects, departments, overrides)
        self.tools_layout = QtWidgets.QVBoxLayout()
        self.tools_chooser = ToolsTable(project=self.project)
        self.stored_widget_map['project'] = self.options_chooser.project_chooser
        self.stored_widget_map['department'] = self.options_chooser.department_chooser
        self.stored_widget_map['overrides'] = self.options_chooser.overrides_chooser

        self.options_chooser.changed_project.connect(self.update_project)
        self.options_chooser.changed_department.connect(self.update_department)
        self.options_chooser.changed_overrides.connect(self.update_overrides)

        self.log_consoles = QtWidgets.QTabWidget()
        self.log_consoles.setFixedHeight(200)
        self.log_consoles.setHidden(True)
        self.log_consoles.setTabsClosable(True)
        self.log_consoles.tabCloseRequested.connect(lambda index: self.close_tab(index))

        self.setLayout(self.layout)
        self.layout.addWidget(self.window_widget)

        self.layout.addLayout(self.content_layout)

        self.content_layout.addWidget(self.options_chooser)
        self.content_layout.addLayout(self.tools_layout)
        self.tools_layout.addWidget(self.tools_chooser)
        self.tools_layout.addStretch(1)
        self.layout.addWidget(self.log_consoles)

        self.update_widgets_from_settings(self.stored_widget_map)

    def update_widgets_from_settings(self, map):
        for name, widget in map.items():
            cls = widget.__class__.__name__
            getter, setter = self.widget_mappers.get(cls, ('getValue', 'setValue'))
            value = self.settings.value(name)
            if setter and value is not None:
                fn = getattr(widget, setter)
                fn(value)  # Set the widget.

    def update_settings_from_widgets(self, map):
        for name, widget in map.items():
            cls = widget.__class__.__name__
            getter, setter = self.widget_mappers.get(cls, ('getValue', 'SetValue'))
            if getter:
                fn = getattr(widget, getter)
                value = fn()
                if value is not None:
                    self.settings.setValue(name, value)  # Set the settings.

        # Notify watcher of changed settings.
        self.settings_changed.emit()


    def close_tab(self, index):

        item = self.log_consoles.widget(index)
        if item.process is None:
            print('task finished, can close')
            self.log_consoles.removeTab(index)

        else:
            reply = QtWidgets.QMessageBox.question(None,
                                                   "Finish Task",
                                                   "This task is still running. Do you want to finish it ?",
                                                   QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No,
                                                   QtWidgets.QMessageBox.No)

            if reply == QtWidgets.QMessageBox.Yes:
                import signal as os_signal

                pid = item.process.processId()
                os.kill(pid, os_signal.CTRL_C_EVENT)
                self.log_consoles.removeTab(index)

        if self.log_consoles.count() == 0:
            self.log_consoles.setHidden(True)
            self.window().setFixedHeight(600)

    def update_project(self, new_project=None, overrides=None):
        print('update project to %s' % new_project)

        self.tools_chooser.update_project(new_project)

    def update_department(self, new_department):
        print('update department to %s' % new_department)
        self.tools_chooser.clean_tools()

        self.current_department = self.options_chooser.department_chooser.getValue()
        self.current_overrides = self.options_chooser.overrides_chooser.getValue()

        self.tools_chooser.read_config(self.current_department, overrides=self.current_overrides)

    def update_overrides(self, overrides):
        print('update overrides to %s' % overrides)
        self.update_department('')

    def close_app(self):
        print('close')
        self.update_settings_from_widgets(self.stored_widget_map)
        self.window().close()


class ToolsLauncherWindow(QtWidgets.QMainWindow):

    def __init__(self, project=None):
        super(ToolsLauncherWindow, self).__init__()
        self.project = project
        self.setWindowFlag(QtCore.Qt.FramelessWindowHint)
        self.setContentsMargins(0, 0, 0, 0)
        self.setMinimumHeight(600)
        self.setFixedWidth(1000)
        self.setWindowTitle('Tools Launcher')

        self.main_widget = ToolsLauncherWidget(project=self.project, parent=self)

        self.setCentralWidget(self.main_widget)

    def update_project(self, new_project):
        self.main_widget.options_chooser(new_project)


def apply_style(app):
    # gradiente_dark = 'qlineargradient(x1: 0, y1: 0, x2: 1, y2: 0, stop: 0 #050505, stop: 1.0 #050590);'
    # gradiente_light = 'qlineargradient(x1: 1, y1: 0, x2: 0, y2: 0, stop: 0 #404060, stop: 1.0 #6060b0);'

    # base_color = (150, 150, 250)
    base_color = (155, 205, 255)
    bg_darker = [i / 12 for i in base_color]
    bg = [i / 9 for i in base_color]
    bg_lighter = [i / 3 for i in base_color]

    default_colors = (50, 150, 250)

    colors = {'Background-darker': 'rgb({}, {}, {})'.format(*bg_darker),
              'Background': 'rgb({}, {}, {})'.format(*bg),
              'Background-lighter': 'rgb({}, {}, {})'.format(*bg_lighter),
              'resalted': 'rgb({}, {}, {})'.format(*default_colors),
              'extra': 'rgb(255, 166, 0)',
              'text': 'rgb(%i, %i, %i)' % (base_color[0], base_color[1], base_color[2]),
              'resalted-line': 'rgb(50, 150, 250)',
              'line': 'rgb(%i, %i, %i)' % (base_color[0], base_color[1], base_color[2])
              }

    path_object = importlib.resources.path(launcher_resources, 'stylesheet.qss')
    with path_object as path:
        with open(path, "r") as f:
            _style = f.read()

    for key, value in colors.items():
        _style = _style.replace('<%s>' % key, value)

    app.setStyleSheet(_style)



if __name__ == '__main__':
    default_project = 'tpt'
    sys.path.insert(0, 'c:/dev')
    app=QtWidgets.QApplication(sys.argv)
    window = ToolsLauncherWindow(project=default_project)
    apply_style(app)
    window.show()
    app.exec()
