import logging
import importlib

from collections import OrderedDict
from pprint import pprint

import shotgrid_lib.database as sg_database
import library.core.config_manager as config_manager

#importlib.reload(config_manager)
#importlib.reload(sg_database)

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

class DependencyTracker():
    def __init__(self,
                 project,
                 pipeline_step,
                 force_status=False,
                 default_variant='Master',
                 database=None,
                 overrides=None):

        self.project = project
        self.pipeline_step = pipeline_step
        self.force_status = force_status
        self.default_variant = default_variant
        self.overrides = overrides if overrides is not None else {}

        logger.debug('Init DependencyTracker')
        logger.debug('======================')

        logger.debug('project: %s' % self.project)
        logger.debug('pipeline_step: %s' % self.pipeline_step)

        logger.debug('force_status: %s' % self.force_status)
        logger.debug('default_variant: %s' % self.default_variant)

        self.database = sg_database.DataBase()
        self.database.fill(self.project, precatch=False)

        logger.debug('precaching Step entity')
        self.database.query_sg_database('Step', as_precache=True)

        self.step_view = self.database['Step'].find_with_filters(code=self.pipeline_step,
                                                                  entity_type=['Asset', 'Shot'],
                                                                  single_item=True)

        self.context_type = self.step_view.entity_type

        logger.debug('precaching Variant entity')

        self.database.query_sg_database('CustomEntity11', as_precache=True)
        self.all_variant_view = self.database['CustomEntity11']
        self.default_variant_view = self.database['CustomEntity11'][self.default_variant]

        self.config_manager = config_manager.ConfigSolver(project=self.project)
        self.step_config = self.config_manager.get_config('%s_dependencies' % self.pipeline_step,
                                                          module='dependency_tracker')
    def get_published_variant(self, item_step_view, step_data, asset_code, geometry_variant, shading_variant):
        if not geometry_variant:
            geometry_variant = 'Master'
        if not shading_variant:
            shading_variant = 'Master'

        step_name = item_step_view.code
        if step_data.get('variant') == 'geometry':
            publish_variant = self.all_variant_view[geometry_variant]
        elif step_data.get('variant') == 'shading':
            publish_variant = self.all_variant_view[shading_variant]
        elif step_data.get('variant') is not None:
            publish_variant = self.all_variant_view[step_data.get('variant')]
        else:
            publish_variant = None

        if not publish_variant or publish_variant.empty:
            logger.error('cant find variant %s' % step_data.get('variant', ''))
            return sg_database.emptyView()

        if self.overrides.get(asset_code, {}).get(step_name):
            published_elements = self.overrides[asset_code][step_name]
        else:
            published_elements = self.get_published_elements(self.asset_view,
                                                             item_step_view,
                                                             step_data,
                                                             publish_variant=publish_variant)

        return published_elements

    def get_asset_dependencies(self,
                               asset_name,
                               geometry_variant='Master',
                               shading_variant='Master',
                               asset_breakdown=None,
                               alias=None):


        self.asset_view = self.database['Asset'][asset_name]
        asset_type = self.asset_view.sg_asset_type
        short_asset_type = asset_type.split(' ')[-1]

        if alias is None:
            asset_code = self.asset_view.code
        else:
            asset_code = alias

        dep_dict = OrderedDict()
        for build_step in self.step_config.get('asset_steps', {}):
            for name, step_data in build_step.items():
                if step_data.get('valid_items') and short_asset_type not in step_data['valid_items']:
                    continue

                step_name = step_data.get('step', name)
                item_step_view = self.database['Step'].find_with_filters(code=step_name,
                                                                         entity_type=['Asset', 'Shot'],
                                                                         single_item=True)
                if item_step_view.empty:
                    continue
                published_elements = self.get_published_variant(item_step_view,
                                                                step_data,
                                                                asset_name,
                                                                geometry_variant,
                                                                shading_variant)
                if published_elements.empty and step_data.get('use_default', False):
                    logger.warning('Can\'t find variant %s %s. Getting master variant' % (geometry_variant, shading_variant))
                    published_elements = self.get_published_variant(item_step_view,
                                                                    step_data,
                                                                    asset_name,
                                                                    'Master',
                                                                    'Master')

                if not published_elements.empty:

                    if not published_elements.sg_asset.empty:
                        if published_elements.sg_asset_name != self.asset_view.code:
                            asset_code = published_elements.sg_asset_name

                    dep_dict[asset_code] = dep_dict.get(asset_code, OrderedDict())
                    if asset_breakdown is None:
                        asset_breakdown = {}

                    dep_dict[asset_code][step_name] = asset_breakdown.copy()
                    dep_dict[asset_code][step_name]['publish'] = published_elements
        self.dependencies_data = dep_dict
        return dep_dict


    def get_shot_breakdown_dependencies(self):
        self.breakdown = None
        self.breakdowns = self.shot_view.sg_breakdowns

        #self.dependencies = self.breakdowns.sg_asset
        default_variant = self.database['CustomEntity11']['Master']

        dep_dict = OrderedDict()
        breakdown_data = {}
        for asset_breakdown in self.breakdowns:
            asset_view = asset_breakdown.sg_asset
            asset_name = asset_view.code
            asset_type = asset_view.sg_asset_type
            if asset_breakdown.sg_alias:
                asset_code = asset_breakdown.sg_alias
            else:
                asset_code = '%s_%03d' % (asset_view.code, asset_breakdown.sg_instance)



            geometry_variant = asset_breakdown.sg_geometry_variant
            shading_variant = asset_breakdown.sg_shading_variant

            if geometry_variant.empty:
                geometry_variant = default_variant
            if shading_variant.empty:
                shading_variant = default_variant

            children = asset_breakdown.sg_breakdowns.code
            if isinstance(children, str):
                children = [children]
            elif not isinstance(children, list):
                    children = []

            asset_breakdown_data = {'instance_number': asset_breakdown.sg_instance,
                                    'asset_name': asset_name,
                                    'asset_type': asset_type,
                                    'geometry_variant': geometry_variant,
                                    'shading_variant': shading_variant,
                                    'children': children
                                    }
            if asset_breakdown.sg_parent_asset:
                asset_breakdown_data['parent_asset'] = asset_breakdown.sg_parent_asset
                asset_breakdown_data['parent_name'] = asset_breakdown.sg_name_in_parent

            if asset_breakdown.sg_shot_type:
                asset_breakdown_data['shot_type'] = asset_breakdown.sg_shot_type
            else:
                asset_breakdown_data['shot_type'] = 'Medium'


            breakdown_data[asset_code] = asset_breakdown_data.copy()
            asset_data = self.get_asset_dependencies(asset_name,
                                                     geometry_variant=geometry_variant.code,
                                                     shading_variant=shading_variant.code,
                                                     alias=asset_code,
                                                     asset_breakdown=asset_breakdown_data
                                                     )

            dep_dict.update(asset_data)


        return dep_dict, breakdown_data

    def get_shot_dependencies(self, shot_name):

        dependencies_dic = {}
        logger.debug('Get shot dependencies')
        logger.debug('=====================')
        logger.debug('shot_name: %s' % shot_name)

        self.shot_view = self.database['Shot'][shot_name]

        assets_data, breakdown_data = self.get_shot_breakdown_dependencies()
        dependencies_dic.update(assets_data)

        for build_step in self.step_config.get('shot_steps', {}):
            for name, step_data in build_step.items():
                step_name = step_data.get('step', name)
                print(step_name)
                item_step_view = self.database['Step'].find_with_filters(code=step_name,
                                                                         entity_type='Shot',
                                                                         single_item=True)

                published_elements = self.get_published_variant(item_step_view,
                                                                step_data,
                                                                self.shot_view.code,
                                                                'Master',
                                                                'Master')
                if not published_elements.empty:
                    dependencies_dic['Shot'] = dependencies_dic.get('Shot', OrderedDict())
                    dependencies_dic['Shot'][step_name] = {'publish': published_elements}
                    dependencies_dic['Shot'][step_name]['breakdown'] = breakdown_data

        self.dependencies_data = dependencies_dic
        return dependencies_dic

    def represent(self):
        for asset_name, asset_dependencies in self.dependencies_data.items():
            logger.info(asset_name)
            for step, publish in asset_dependencies.items():
                logger.info('\t %s' % step)
                if not publish or publish.empty:
                    logger.info('\t\t not found')
                    continue
                logger.info('\t\t%s, %s, %s, %s' % (publish.code, publish.sg_version_number, publish.sg_variant_name, publish.sg_status_list))

    def as_dictionary(self, items=None):
        if items is None:
            items = self.dependencies_data

        dependency_data = OrderedDict()
        logger.info('Precache breakdown dependencies')
        for alias, alias_data in items.items():
            dependency_data[alias] = OrderedDict()
            for step_name, item_data in alias_data.items():
                publish_data = item_data['publish']

                build_data = {'files': publish_data.sg_files,
                              'published_folder': publish_data.sg_published_folder,
                              'step': publish_data.sg_step_name,
                              'version_number': publish_data.sg_version_number,
                              'id': publish_data.id,
                              'hash': publish_data.sg_hash,
                              'variant': publish_data.sg_variant_name,
                              'status': publish_data.sg_status_list,

                              }

                if publish_data.sg_asset:
                    instance_number = item_data.get('instance_number', 0)
                    parent_asset = item_data.get('parent_asset', '')
                    name_in_parent = item_data.get('parent_name', '')
                    shot_type = item_data.get('shot_type', 'Medium')
                    build_data['code'] = publish_data.sg_asset.code
                    build_data['asset_type'] = publish_data.sg_asset.sg_asset_type
                    build_data['alias'] = alias
                    build_data['parent'] = parent_asset
                    build_data['name_in_parent'] = name_in_parent
                    build_data['shot_type'] = shot_type

                    build_data['children'] = item_data.get('children', [])

                    build_data['instance'] = instance_number

                else:
                    build_data['code'] = publish_data.sg_context.code
                    build_data['breakdown'] = item_data['breakdown']

                dependency_data[alias][publish_data.sg_step_name] = build_data

        return dependency_data


    def get_search_context(self, asset_view, step_data):

        context = step_data.get('context', '')
        if context == 'Shot':
            search_context = self.shot_view
        elif context == 'Asset':
            search_context = asset_view

        elif context.find('.') > -1:
            context, attribute = context.split('.', 1)
            if context == 'Shot':
                search_context = self.shot_view.get_field_value(attribute, unique=True)
            elif context == 'Asset':
                search_context = asset_view.get_field_value(attribute, unique=True)

            elif context == 'CustomEntity12':
                if len(step_data['valid_items']) == 1:
                    asset_type = step_data['valid_items'][0]
                    search_context = self.shot_view.sg_breakdowns.sg_asset.find_with_filters(sg_asset_type=asset_type,
                                                                                             single_item=True,
                                                                                             )

            else:
                search_context = self.database[context][attribute]



        elif context != '':
            filters = [['code', 'is', context]]
            self.database.query_sg_database('Asset', filters=filters)
            search_context = self.database['Asset'][context]
        else:
            search_context = asset_view

        return search_context

    def query_alternative_item(self, item, item_step_view, step_data):
        alt_step_data = step_data.copy()
        alt_publish_data = step_data['alt_publish'].copy()
        for key, value in alt_publish_data.items():
            alt_step_data[key] = value

        if 'alt_publish' not in alt_publish_data:
            alt_step_data['alt_publish'] = None

        logger.warning('published element not found, doing recursive search')
        context = alt_step_data.get('context')
        if context != 'Asset':
            context = 'Shot'

        this_item_step_view = self.database['Step'].find_with_filters(code=alt_step_data['step'],
                                                                      entity_type=context,
                                                                      single_item=True)

        published_elements = self.get_published_elements(item, this_item_step_view, alt_step_data)

        return published_elements

    def get_published_elements(self, asset_view, item_step_view, step_data, publish_variant=None):

        logger.info('%s %s - %s %s' % ('=' * 2, asset_view.code, step_data['step'], '=' * 2))
        #asset_view.precache_dependencies(fields=['sg_published_elements'])
        search_context = self.get_search_context(asset_view, step_data)
        item_step_view = self.database['Step'][step_data['step']]
        if not search_context.empty and not search_context.sg_published_elements.empty:
            if publish_variant :
                published_elements = search_context.sg_published_elements.find_with_filters(sg_step=item_step_view,
                                                                                            sg_variant=publish_variant,
                                                                                            sg_complete=True,
                                                                                            sg_delete=False)
                if published_elements.empty and step_data.get('use_default_variant', False):
                    logger.warning('Can\'t find the variant %s, quering for the default variant' % publish_variant.code)

                    published_elements = search_context.sg_published_elements.find_with_filters(sg_step=item_step_view,
                                                                                                sg_variant=self.default_variant_view,
                                                                                                sg_complete=True,
                                                                                                sg_delete=False)
            else:

                published_elements = search_context.sg_published_elements.find_with_filters(sg_step=item_step_view,
                                                                                            sg_complete=True,
                                                                                            sg_delete=False)
        else:
            published_elements = sg_database.emptyView()

        if published_elements.empty and step_data.get('alt_publish'):
            published_elements = self.query_alternative_item(asset_view, item_step_view, step_data)

        recomended_version = self.filter_by_status_list(published_elements, step_data.get('status'))

        return recomended_version

    def filter_by_status_list(self, published_elements, status_list):
        if published_elements.empty:
            return published_elements

        if not self.force_status or status_list is None:
            return max(published_elements)

        if isinstance(status_list, str):
            status_list = [status_list]

        for status in status_list:
            filtered = published_elements.find_with_filters(sg_status_list=status)
            if not filtered.empty:
                return max(filtered)

        return sg_database.emptyView()

    def filter_valid_items(self, item_type_list, item_step_type):
        if item_step_type == 'Shot':
            if item_type_list == 'context':
                return self.context_view

        elif item_step_type == 'Asset':
            items = self.dependencies.find_with_filters(sg_asset_type=item_type_list)
            return items

    def get_dependencies(self, context, geometry_variant='Master', shading_variant='Master'):

        filters = [['code', 'is', context],
                   ['project.Project.code', 'is', self.project]]
        self.database.query_sg_database(self.context_type, filters=filters, as_precache=False)
        self.entity_view = self.database[self.context_type][context]


        if self.context_type == 'Shot':
            self.entity_view.sg_breakdowns.precache_dependencies(fields=['sg_asset'])
            self.entity_view.sg_breakdowns.sg_asset.precache_dependencies(fields=['sg_published_elements'])

            dependencies = self.get_shot_dependencies(context)
        elif self.context_type == 'Asset':
            self.entity_view.precache_dependencies(fields=['sg_published_elements'])

            dependencies = self.get_asset_dependencies(context,
                                                       geometry_variant=geometry_variant,
                                                       shading_variant=shading_variant)
        else:
            dependencies = OrderedDict()

        return self.entity_view, dependencies

if __name__ == '__main__':
    dependency_tracker = DependencyTracker('tpt', 'AssetAnimation', force_status=False, database=None)
    dependency_tracker.get_dependencies('LeoHero')
    #dependency_tracker.get_dependencies('s00_ep00_sq010_sh010')
    pprint(dependency_tracker.as_dictionary())
    #dependency_tracker.represent()