#!/usr/bin/env python3

from __future__ import absolute_import

import importlib
import os
import re
import sys
import library
print(library)

from Deadline.Plugins import DeadlinePlugin, PluginType
from Deadline.Scripting import RepositoryUtils, StringUtils
from FranticX.Processes import ManagedProcess
from System.IO import Path
from System.Text.RegularExpressions import Regex

import launcher.lib.set_environment as environment_solver

importlib.reload(environment_solver)

######################################################################
# This is the function that Deadline calls to get an instance of the
# main DeadlinePlugin class.
######################################################################
def GetDeadlinePlugin():
    return AACommandLinePlugin()


def CleanupDeadlinePlugin( deadlinePlugin ):
    deadlinePlugin.Cleanup()


######################################################################
## This is the main DeadlinePlugin class for the CommandLine plugin.
######################################################################
class AACommandLinePlugin( DeadlinePlugin ):
    def __init__( self ):
        # type: () -> None
        if sys.version_info.major == 3:
            super().__init__()
        self.InitializeProcessCallback += self.InitializeProcess
        self.RenderTasksCallback += self.RenderTasks
        self.ShProcess = None
    
    def Cleanup( self ):
        # type: () -> None
        for stdoutHandler in self.StdoutHandlers:
            del stdoutHandler.HandleCallback

        del self.InitializeProcessCallback
        del self.RenderTasksCallback

        if self.ShProcess:
            self.ShProcess.Cleanup()
            del self.ShProcess
    

    def set_environment(self):


        job = self.GetJob()

        keys = job.GetJobEnvironmentKeys()
        project_name = self.GetPluginInfoEntryWithDefault( "Project", "" )

        if not project_name:
            self.LogWarning('Can\'t find project env var in this job')
            return
        
        self.LogInfo('Getting  project %s environement' % project_name)

        self.project_environment = environment_solver.ProjectEnvironment(project=project_name)
        
        overrides = {}
        for key in keys:
            overrides[key] = self.GetProcessEnvironmentVariable(key)

        environment = self.project_environment.get_environment(project_name, overrides=overrides)

        for key, value in environment.items():
            if not value:
                continue

            elif len(value) == 1:
                env_var_value = value[0]
                env_var_value = str(env_var_value)
                #if env_var_value.find(' ') > -1:
                #    env_var_value = '"%s"' % env_var_value
                                    
            else:
                value = [str(val) for val in value if val is not None]
                env_var_value = ';'.join(value)
            if isinstance(env_var_value, str):
                env_var_value = env_var_value.replace('\\', '/')
                
            env_var_name = key.upper()

            #self.LogInfo('Setting environment variable %s to %s' % (env_var_name, env_var_value))
            self.SetProcessEnvironmentVariable(env_var_name, env_var_value)


    def InitializeProcess( self ):
        # type: () -> None
        self.SingleFramesOnly = self.GetBooleanPluginInfoEntryWithDefault( "SingleFramesOnly", False )
        self.LogInfo( "Single Frames Only: %s" % self.SingleFramesOnly )

        self.PluginType = PluginType.Advanced
        self.StdoutHandling = True
        self.set_environment()
        self.AddStdoutHandlerCallback( ".*Progress: (\d+)%.*" ).HandleCallback += self.HandleProgress

    def RenderTasks( self ):
        # type: () -> None
        executable = self.CommandLineGetRenderExecutable()
        arguments = self.GetRenderArguments()
        startupDir = self.GetStartupDirectory()


        standalone = self.GetPluginInfoEntryWithDefault( 'Standalone', 'cmd' )
        if standalone != 'cmd':
            project_name = self.GetPluginInfoEntryWithDefault( "Project", "" )
            tools_data = self.project_environment.get_project_tools(project_name).get(standalone)
            if tools_data:
                tools_data['name'] = standalone
                standalone_command, standalone_environment = self.project_environment.get_launch_data(tools_data)

                arguments = '"{}" {}'.format(executable, arguments) 
                executable = standalone_command.replace('\\', '/')

                self.LogInfo( "New executable : %s" % executable )
                self.LogInfo( "New arguments : %s" % arguments )
                


        shellExecute = self.GetBooleanPluginInfoEntryWithDefault( "ShellExecute", False )
        self.LogInfo( "Execute in Shell: %s" % shellExecute )

        if not shellExecute:
            self.LogInfo( "Invoking: Run Process" )
            # StdOut/Err will NOT be captured here as unmanaged process
            exitCode = self.RunProcess( executable, arguments, startupDir, -1 )
        else:
            self.LogInfo( "Invoking: Managed Shell Process" )
            self.ShProcess = ShellManagedProcess( self, arguments, startupDir ) # type: ignore
            self.RunManagedProcess( self.ShProcess )
            exitCode = self.ShProcess.ExitCode # type: ignore

        self.LogInfo( "Process returned: %s" % exitCode )

        if exitCode != 0:
            self.FailRender( "Process returned non-zero exit code '{}'".format( exitCode ) )

    def CommandLineGetRenderExecutable( self ):
        # type: () -> str
        executable = RepositoryUtils.CheckPathMapping( self.GetPluginInfoEntryWithDefault( "Executable", "" ).strip() )
        if executable != "":
            self.LogInfo( "Executable: %s" % executable )
        return executable
    
    def GetRenderArguments( self ):
        # type: () -> str
        arguments = RepositoryUtils.CheckPathMapping( self.GetPluginInfoEntryWithDefault( "Arguments", "" ).strip() )

        arguments = re.sub( r"<(?i)STARTFRAME>", str( self.GetStartFrame() ), arguments )
        arguments = re.sub( r"<(?i)ENDFRAME>", str( self.GetEndFrame() ), arguments )
        arguments = re.sub( r"<(?i)QUOTE>", "\"", arguments)

        arguments = self.ReplacePaddedFrame( arguments, "<(?i)STARTFRAME%([0-9]+)>", self.GetStartFrame() )
        arguments = self.ReplacePaddedFrame( arguments, "<(?i)ENDFRAME%([0-9]+)>", self.GetEndFrame() )

        count = 0
        for filename in self.GetAuxiliaryFilenames():
            localAuxFile = Path.Combine( self.GetJobsDataDirectory(), filename )
            arguments = re.sub( r"<(?i)AUXFILE" + str( count ) + r">", localAuxFile.replace( "\\", "/" ), arguments )
            count += 1

        arguments = arguments.replace(u'\u201c', '"').replace(u'\u201d', '"')

        if arguments != "":
            self.LogInfo( "Arguments: %s" % arguments )
        
        return arguments

    def GetStartupDirectory( self ):
        # type: () -> str
        startupDir = self.GetPluginInfoEntryWithDefault( "StartupDirectory", "" ).strip()
        if startupDir != "":
            self.LogInfo( "Startup Directory: %s" % startupDir )
        return startupDir
    
    def ReplacePaddedFrame( self, arguments, pattern, frame ):
        # type: (str, str, int) -> str
        frameRegex = Regex( pattern )
        while True:
            frameMatch = frameRegex.Match( arguments )
            if frameMatch.Success:
                paddingSize = int( frameMatch.Groups[ 1 ].Value )
                if paddingSize > 0:
                    padding = StringUtils.ToZeroPaddedString( frame, paddingSize, False )
                else:
                    padding = str(frame)
                arguments = arguments.replace( frameMatch.Groups[ 0 ].Value, padding )
            else:
                break
        
        return arguments

    def HandleProgress( self ):
        # type: () -> None
        progress = float( self.GetRegexMatch(1) )
        self.SetProgress( progress )

#################################################################################
## This is the shell managed process for running SHELL commands.
#################################################################################
class ShellManagedProcess( ManagedProcess ):
    '''
    This class provides a Deadline Managed Process using a pre-selected shell executable and provided command/arguments
    '''
    
    def __init__( self, deadlinePlugin, argument, directory ):
        # type: (CommandLinePlugin, str, str) -> None
        if sys.version_info.major == 3:
            super().__init__()
        self.deadlinePlugin = deadlinePlugin
        self.Argument = argument
        self.Directory = directory
        self.ExitCode = -1
        
        self.InitializeProcessCallback += self.InitializeProcess
        self.RenderExecutableCallback += self.RenderExecutable
        self.RenderArgumentCallback += self.RenderArgument
        self.StartupDirectoryCallback += self.StartupDirectory
        self.CheckExitCodeCallback += self.CheckExitCode

    def Cleanup( self ):
        # type: () -> None
        for stdoutHandler in self.StdoutHandlers:
            del stdoutHandler.HandleCallback

        del self.InitializeProcessCallback
        del self.RenderExecutableCallback
        del self.RenderArgumentCallback
        del self.StartupDirectoryCallback
        del self.CheckExitCodeCallback
    



    def InitializeProcess( self ):
        # type: () -> None
        self.PopupHandling = True
        self.StdoutHandling = True
        self.HideDosWindow = True
        # Ensure child processes are killed and the parent process is terminated on exit
        self.UseProcessTree = True
        self.TerminateOnExit = True

        self.ShellString = self.deadlinePlugin.GetPluginInfoEntryWithDefault( "Shell", "default" )
        self.AddStdoutHandlerCallback( ".*Progress: (\d+)%.*" ).HandleCallback += self.HandleProgress
    


    def RenderExecutable( self ):
        # type: () -> str
        if self.ShellString == "default":
            # Grab the default shell executable from the User's environment.
            shellExecutable = os.environ.get( "SHELL", "/bin/sh" )
        else:
            # Search the list of potential paths for the shell executable.
            shellExecutable = self.deadlinePlugin.GetRenderExecutable("ShellExecutable_" + self.ShellString, self.ShellString)
        
        return shellExecutable

    def RenderArgument( self ):
        # type: () -> str
        if self.ShellString == "cmd":
            return '/c "{}"'.format( self.Argument )
        else:
            # Since we're surrounding the command in double quotes, we need to escape the ones already there.
            # Literal quotes need special treatment; we'll deal with these later.
            escapedQuotes = self.Argument.replace(r'\"','<QUOTE>') 
            # The most universal way to escape nested quotes in a shell command is to first end
            # the 'current' quote block, escape the quote character, and re-start another block
            escapedQuotes = escapedQuotes.replace(r'"', r'"\""')
            # Doing the above on a \" string would result in \"\""; 
            # this mucks with any shells that *do* support escaping a " within a block using \" (e.g., bash),
            # so we move the backslash out of the block and escape it as well.
            escapedQuotes = escapedQuotes.replace('<QUOTE>', r'"\\\""')
            return '-c "{}"'.format( escapedQuotes )

    def StartupDirectory( self ):
        # type: () -> str
        return self.Directory

    def CheckExitCode( self, exitCode ):
        # type: (int) -> None
        self.ExitCode = exitCode

    def HandleProgress(self):
        # type: () -> None
        progress = float( self.GetRegexMatch(1) )
        self.deadlinePlugin.SetProgress( progress )
