import logging
import importlib
import os

import packages_io

import library.core.general_exceptions as general_exceptions
import library.core.config_manager as config_manager
import library.core.dynamic_run as dynamic_run
import library.core.references_solver as references_solver
import library.sandbox as sandbox_builder

import dependency_tracker.lib.dependency_tracker as dependency_tracker
import builder.lib.helpers as builder_helpers
importlib.reload(builder_helpers)

logger = logging.getLogger(__name__)


class AssetBuilder(references_solver.ReferencesSolver):
    def __init__(self,
                 pipeline_step,
                 project='TPT',
                 database=None,
                 output_path=None,
                 overrides=None
                 ):

        super().__init__(self)

        self.project = project
        self.default_variant = None
        self.output_path = output_path
        self.overrides = overrides
        self.pipeline_step = pipeline_step.split('_')[0]
        self.database = database

        self.current_app = packages_io.get_application()
        self.full_version_path = ''

        self.config_solver = config_manager.ConfigSolver(project=self.project)
        self.project_data = self.config_solver.get_config('project')
        self.resolutions_data = self.config_solver.get_config('show_resolutions')

        self._vars_to_solve = {}
        self.global_builder_data = self.config_solver.get_config('global_builder', module='builder')

        self.step_builder_data = self.config_solver.get_config('definitions/%s' % pipeline_step, module='builder')

        if not self.step_builder_data:
            logger.error('Can\'t find definition for pipeline step %s' % self.pipeline_step)
            raise general_exceptions.ConfigNotFound('definitions/%s' % self.pipeline_step, 'builder')

        self.step_build_data = self.step_builder_data
        self.sandbox_folder = self.step_builder_data['sandbox_folder']

        self.extension = self.step_builder_data.get('extension', 'mb')

        self.dependencies_solver = dependency_tracker.DependencyTracker(self.project,
                                                                        self.pipeline_step,
                                                                        overrides= self.overrides)
        self.context_type = self.dependencies_solver.context_type

    def set_context(self, entity_name, variant='Master'):
        self.entity_name = entity_name
        self.variant = variant
        logger.debug('variant: %s'% variant)

        logger.debug('Setting context to Entity name: %s' % entity_name)
        logger.debug('With variant: %s' % self.variant)

        self.entity_view, _ = self.dependencies_solver.get_dependencies(self.entity_name, geometry_variant=variant, shading_variant=variant)
        self.dependencies = self.dependencies_solver.as_dictionary()

        return self.dependencies_solver

    def launch_pre_build_tasks(self):
        logger.debug('Launch pre build tasks')
        packages_io.file_new()
        sandbox_solver = sandbox_builder.Sandbox(project=self.project)

        if self.entity_view.type == 'Asset':
            sandbox_solver.set_context(asset_name=self.entity_view.code,
                                       asset_type=self.entity_view.sg_asset_type)
        else:
            sandbox_solver.set_context(shot_name=self.entity_view.code)

        sandbox_path = sandbox_solver.build()
        logger.debug('Created sandbox to: %s' % sandbox_path)
        if not self.variant:
            self.variant = 'Master'

        sandbox_solver.content = self.pipeline_step
        sandbox_solver.variant = self.variant
        sandbox_solver.extension = self.extension

        filename = sandbox_solver.generate_filename()
        sandbox_solver.generate_path()
        if self.output_path is None:
            full_path = os.path.join(sandbox_path, self.sandbox_folder, filename)
        else:
            full_path = self.output_path
        full_version_path = sandbox_builder.get_version_path(full_path, extension=self.extension)

        self.set_scene_settings()

        builder_helpers.create_template(self.pipeline_step,
                                        project=self.project,
                                        asset_type=self.entity_view.sg_asset_type)

        logger.info('Set filename to: %s ' % full_version_path)
        packages_io.file_save(rename_to=full_version_path)

        self.full_version_path = full_version_path

    def set_scene_settings(self):
        if self.context_type == 'Shot':
            shot_resolution = self.entity_view.sg_resolution
            start_frame = int(self.entity_view.sg_cut_in)
            end_frame = int(self.entity_view.sg_cut_out)
            if not shot_resolution:
                shot_resolution = self.entity_view.sg_sequence.sg_resolution
                if not shot_resolution:
                    shot_resolution = self.entity_view.sg_sequence.episode.sg_resolution
                    if not shot_resolution:
                        shot_resolution = 'Default'
        else:
            shot_resolution = 'Default'
            start_frame = 1001
            end_frame = 1048

        shot_data = self.resolutions_data.get(shot_resolution)


        if not shot_data:
            shot_data = self.resolutions_data.get('Default')

        self.shot_data = shot_data
        shot_data['context_type'] = self.context_type
        if self.context_type == 'Shot':
            shot_data['start_frame'] = start_frame
            shot_data['end_frame'] = end_frame
        packages_io.set_scene_settings(shot_data)

    def launch_task(self, step_data):
        step_name = step_data['step']
        code = step_data['code']
        logger.info('== Launching build step %s for asset: %s ==' % (step_name, code))

        asset_type = step_data.get('asset_type', 'Shot').split(' ')[-1]
        function_data = self.step_build_data.get('steps', {}).get(step_name, {}).get(asset_type, {})
        if not function_data:
            function_data = self.step_build_data.get('steps', {}).get(step_name, {}).get('All', {})

        if not function_data:
            function_data = self.global_builder_data.get(step_name, {})

        if not function_data:
            return

        if step_data['step'] in self.overrides:
            step_override = self.overrides[step_data['step']]
            if 'version' in step_override:
                step_data['version_number'] = int(step_override['version'])

            if 'variant' in step_override:
                step_data['variant'] = step_override['variant']

        function_data['args'] = function_data.get('args', {})
        function_data['args']['step_data'] = step_data
        function_data['args']['project'] = self.project
        function_data['args']['current_step'] = self.pipeline_step
        function_data['args']['filename'] = self.full_version_path
        logger.debug('Function: %s' % function_data.get('function'))
        logger.debug('Module: %s' % function_data.get('module'))
        output = dynamic_run.run_scripts(function_data)

        return output


    def get_child_asset(self, parent_asset):
        logger.info('Get children assets')
        children = []

        for breakdown_view in self.entity_view.sg_breakdowns:
            asset_maya_name = breakdown_view.sg_name_in_parent
            children.append(asset_maya_name)
        logger.debug('Children: %s' % ', '.join(children))

        return children

    def launch_build_tasks(self):

        for pre_task_name, pretask_data in self.step_build_data.get('pretask', {}).items():
            step_data = {'code': self.entity_view.code,
                         'step': self.pipeline_step}

            pretask_data['args'] = pretask_data.get('args', {})
            pretask_data['args']['step_data'] = step_data
            pretask_data['args']['project'] = self.project
            output = dynamic_run.run_scripts(pretask_data)

        for asset_code, asset_data in self.dependencies.items():
            for step_name, step_data in asset_data.items():
                if not self.add_last_version and step_name == self.pipeline_step:
                    continue
                self.launch_task(step_data)

    def set_cameras_apperture(self):
        if not hasattr(packages_io, 'set_cameras_data'):
            return
        if not self.shot_data:
            return
        packages_io.set_cameras_data(self.shot_data)

    def launch_post_build_tasks(self):
        self.set_cameras_apperture()
        if self.current_app in ['maya', 'mayapy'] and self.context_type == 'Shot':
            import maya.cmds as cmds
            start_frame = int(self.entity_view.sg_cut_in)
            start_animation = start_frame - 10

            end_frame = int(self.entity_view.sg_cut_out)
            end_animation = end_frame + 10

            logger.info('Set frame range to %s %s' % (start_frame, end_frame))

            cmds.playbackOptions(ast=start_animation, e=True)
            cmds.playbackOptions(aet=end_animation, e=True)
            cmds.playbackOptions(minTime=start_frame, e=True)
            cmds.playbackOptions(maxTime=end_frame, e=True)
            cmds.currentTime(start_frame, e=True)

        packages_io.file_save(rename_to=self.full_version_path)

    def build(self, add_last_version=False, overrides=None, dependencies=None):
        errors = False
        self.overrides = overrides
        if overrides:
            for alias, alias_data in overrides.items():
                if alias not in self.dependencies:
                    continue
                for step_name, step_data in alias_data.items():
                    if step_name not in self.dependencies[alias]:
                        continue
                    if not step_data.get('enabled'):
                        self.dependencies[alias].pop(step_name, None)
                        continue
                    self.dependencies[alias][step_name]['publish'] = step_data['publish']
                    self.dependencies[alias][step_name]['variant'] = step_data['publish'].sg_variant_name
                    self.dependencies[alias][step_name]['version_number'] = step_data['publish'].sg_version_number
                    self.dependencies[alias][step_name]['published_folder'] = step_data['publish'].sg_published_folder
                    self.dependencies[alias][step_name]['status'] = step_data['publish'].sg_status_list
                    self.dependencies[alias][step_name]['files'] = step_data['publish'].sg_files
                    self.dependencies[alias][step_name]['hash'] = step_data['publish'].sg_hash
                    self.dependencies[alias][step_name]['files'] = step_data['publish'].sg_files
                    self.dependencies[alias][step_name]['files'] = step_data['publish'].sg_files

        logger.info('==== Start build ====')
        self.add_last_version = add_last_version
        logger.debug('Add last version: %s' % add_last_version)

        logger.info('== Running pre build tasks ==')
        self.launch_pre_build_tasks()

        logger.info('== Running build tasks ==')
        self.launch_build_tasks()

        logger.info('== Running post build tasks ==')
        self.launch_post_build_tasks()
        logger.info('==== Builder finished ====')

        return errors


if __name__ == '__main__':

    path = 'C:/dev/builder/config/modeling_builder.yaml'
    builder = AssetBuilder('Modeling', path)
