import os
import logging

from PySide import QtWidgets

import packages_io

import library.ui.context_widget as context_widget
import library.ui.logging_dialog as logging_dialog
import library.ui.dialog_widgets as dialog_widgets

import builder.lib.builder_core as builder_core
import builder.lib.dependencies_widget as dependencies_widget


logger = logging.getLogger(__name__)
# logger.setLevel(logging.INFO)

class BuilderWindow(packages_io.MainWindow):
    """
    Main builder for all the pipeline steps

    pipeline_step: The type of the publish to be created
    project: The code of the Shotgrid project to work on it

    """
    def __init__(self, pipeline_step='Model', project=None):
        self.pipeline_step = pipeline_step

        self.title = 'Builder: %s' % pipeline_step
        super( BuilderWindow, self ).__init__(self.title)

        self.setContentsMargins(6, 6, 6, 6)

        self.main_widget = BuilderWidget(self.pipeline_step, project=project, parent=self)
        self.apply_style(self)
        self.setMinimumSize(500, 400)
        self.setCentralWidget(self.main_widget)
        QtWidgets.QApplication.processEvents()

        self.main_widget.start()

    def closeEvent(self, event):
        self.main_widget.close_window()
        event.accept()

class BuilderWidget(context_widget.ContextWidget):

    def __init__(self, pipeline_step, parent=None, project=None):
        super(BuilderWidget, self).__init__(pipeline_step, parent=parent, project=project)
        self.setObjectName('Builder: %s' % pipeline_step)
        self.pipeline_step = pipeline_step
        self.definition_apply_latest = self.definition.get('apply_latest', True)
        self.log_dialog = None
        self.project = project
        if self.definition_apply_latest:
            self.add_last_version_widget(self.main_layout)

        self.builder_core = builder_core.AssetBuilder(self.pipeline_step, project=self.project)

        self.add_dependencies_widget(self.main_layout)
        self.task_changed.connect(self.update_dependencies)
        self.last_task_code = None
        self.start_widget = None
        self.end_widget = None

        self.app_layout.addLayout(self.main_layout)
        self.app_layout.addStretch(1)

        self.launch = QtWidgets.QPushButton('Build')
        self.launch.setFixedHeight(30)
        self.app_layout.addWidget(self.launch)
        self.launch.clicked.connect(self.launch_accept)
        self.launch.setEnabled(True)

    def update_dependencies(self):
        task_name = self.task_widget.getValue()
        if not task_name or task_name == self.last_task_code:
            return
        else:
            self.last_task_code = task_name
        entity_data = self.context_widget.getValue()

        if 'name' in entity_data:
            if not entity_data['name']:
                return
            entity_name = entity_data['name']
        else:
            if not entity_data['shot']:
                return

            entity_name = entity_data['shot']

        step_view = self.database['Step'][self.pipeline_step]
        context_type = step_view.entity_type

        entity_view = self.database[context_type][entity_name]

        task_view = self.database['Task'].find_with_filters(content=task_name,
                                                            entity=entity_view,
                                                            single_item=True)
        if task_view.empty:
            return

        variant_name = task_view.sg_variant_name

        dependencies = self.builder_core.set_context(entity_name, variant=variant_name)

        if context_type == 'Shot':
            asset_type = None
        else:
            asset_type = entity_view.sg_asset_type

        self.dependencies_widget.update_model(dependencies)

    def add_dependencies_widget(self, layout):

        self.dependencies_widget = dependencies_widget.DependenciesTreeView()
        layout.addWidget(self.dependencies_widget)

    def launch_accept(self):
        self.open_log_window()

        if self.definition_apply_latest:
            add_latest_version = self.last_version_widget.getValue()
        else:
            add_latest_version = False

        dependencies = self.dependencies_widget.getValue()
        errors = self.builder_core.build(add_last_version=add_latest_version, overrides=dependencies)

        task_name = self.task_widget.getValue()
        current_task = self.tasks_view.find_with_filters(content=task_name,single_item=True)
        current_task.sg_status_list = 'ip'
        current_task.update_shotgun()


        if not errors:
            dlg = dialog_widgets.CustomDialog(self, 'Done!!', 'The builder finished correctly')
        else:
            dlg = dialog_widgets.CustomDialog(self, 'ERROR!!', 'Some builds didn\'t finished correctly')
        dlg.exec_()


    def close_window(self):
        if self.log_dialog:
            self.log_dialog.close()
        self.parent().close()

    def open_log_window(self):
        if self.log_dialog is None:
            self.log_dialog = logging_dialog.LoggingDialog(self, 'Builder: %s' % self.pipeline_step)

        if not self.log_dialog.isVisible():
            self.log_dialog.show()
            self.log_dialog.init_handler()

def open_builder(pipeline_step='', project=None):
    import os
    builder_window = BuilderWindow(pipeline_step, project=project)
    builder_window.show()


def open_builder_substance(task_view, project):

    entity = task_view.entity
    pipeline_step = task_view.step_name

    os.environ['PIPE_CONTEXT'] = 'Asset'
    os.environ['PIPE_ASSET_NAME'] = entity.code

    os.environ['PIPE_ASSET_TYPE'] = entity.sg_asset_type

    builder_window = BuilderWindow(pipeline_step, project=project)
    builder_window.setObjectName('Builder window')

    builder_window.show()
    builder_window.start()


