import requests

from PySide import QtWidgets, QtGui, QtCore

import shotgrid_lib.database as database


class SubClassThread(QtCore.QThread):

    def __init__(self, widget, parent=None):
        super(SubClassThread, self).__init__(parent)
        self.widget = widget

    def run(self):
        self.widget.background_thread()

class AssetStatusBar(QtWidgets.QFrame):
    def __init__(self, parent=None, color=QtGui.QColor("green")):
        super(AssetStatusBar, self).__init__(parent)
        self.setFrameShape(QtWidgets.QFrame.HLine)
        self.setFrameShadow(QtWidgets.QFrame.Plain)
        self.setFixedHeight(3)
        self.setLineWidth(10)
        self.setMidLineWidth(10)

        self.setContentsMargins(0, 0, 0, 0)
        self.setColor(color)

    def setColor(self, color):
        if isinstance(color, str):
            color = QtGui.QColor(color)
        pal = self.palette()
        pal.setColor(QtGui.QPalette.WindowText, color)
        self.setPalette(pal)


class EntityCard(QtWidgets.QFrame):
    width = 250
    height = 50
    status_bar_height = 5

    clicked = QtCore.Signal(database.View)

    entity_tye = None

    def __init__(self,
                 entity_view,
                 manager=None,
                 card_list=None):
        super().__init__(parent=card_list)
        self.entity_view = entity_view
        self.manager = manager
        self.card_list = card_list
        self.database = self.manager.database

        self.setAcceptDrops(True)
        self.setObjectName('no_squared_light')

        self.thumbnail_height = self.height - self.status_bar_height
        self.thumbnail_width = self.thumbnail_height * 16.0 / 9.0

        self.setFixedSize(self.width, self.height)

        self.main_layout = QtWidgets.QVBoxLayout()

        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        self.data_layout = QtWidgets.QHBoxLayout()
        self.data_layout.setSpacing(0)
        self.data_layout.setContentsMargins(0, 0, 0, 0)

        self.thumbnail_widget = QtWidgets.QPushButton()
        self.thumbnail_widget.setFixedSize(self.thumbnail_width, self.thumbnail_height)
        self.thumbnail_widget.setObjectName('no_squared_light')
        self.data_layout.addWidget(self.thumbnail_widget)
        self.add_info(self.data_layout)
        self.main_layout.addLayout(self.data_layout)
        self.status_widget = AssetStatusBar(parent=self)
        self.main_layout.addWidget(self.status_widget)
        self.setLayout(self.main_layout)

        self.update_widgets = SubClassThread(self)
        self.draw_status()

    def add_info(self, data):
        pass

    def draw_status(self):

        if self.entity_view.empty:
            self.setHidden(True)
            return

        else:
            self.setHidden(False)

        if 'sg_status_list' not in self.entity_view._keys:
            self.status_widget.setColor('blue')
            return

        if self.entity_view.sg_status_list == 'cmpt':
            self.status_widget.setColor('green')
        elif self.entity_view.sg_status_list == 'ip':
            self.status_widget.setColor('lightblue')
        elif self.entity_view.sg_status_list == 'wtg':
            self.status_widget.setColor('yellow')
        else:
            self.status_widget.setColor('orange')


    def background_thread(self):
        self.update()

    def update(self):
        pixmap = self.manager.cached_images.get(self.entity_view.id, None)
        if not pixmap:
            thumbnail_path = self.entity_view.image
            if thumbnail_path is None or thumbnail_path == 'None' or isinstance(thumbnail_path, database.emptyView):
                return
            image = QtGui.QImage()
            image.loadFromData(requests.get(thumbnail_path).content)
            pixmap = QtGui.QPixmap.fromImage(image)
            image_size = QtCore.QSize(self.thumbnail_width, self.thumbnail_height)
            pixmap = pixmap.scaled(image_size)#, aspectRatioMode=QtCore.Qt.KeepAspectRatio)
            self.manager.cached_images[self.entity_view.id] = pixmap
        self.thumbnail_widget.setIcon(pixmap)
        self.thumbnail_widget.setIconSize(pixmap.size())

    def launch_task(self):
        self.clicked.emit(self.entity_view)

    def add_mime_data(self, format, string, mime_item):
        app_prefix = 'application/shotgrid'
        encode_string = QtCore.QByteArray(str(string).encode())
        mime_format = '%s/%s' % (app_prefix, format)
        mime_item.setData(mime_format, encode_string)

    def add_entity_mime_data(self):
        mime = QtCore.QMimeData()
        mime.setText(self.code)
        files = self.sg_files

        full_path = '%s/%s' % (self.sg_published_folder, files['abc_high'])
        full_path = full_path.replace('\\', '/')
        mime.setUrls([full_path])

        self.add_mime_data('code', self.code, mime)
        self.add_mime_data('version_number', str(self.sg_version_number), mime)
        self.add_mime_data('asset_name', self.sg_asset_name, mime)
        self.add_mime_data('step', self.sg_step_name, mime)
        self.add_mime_data('variant', self.sg_variant_name, mime)
        self.add_mime_data('hash', self.sg_hash, mime)
        self.add_mime_data('project', self.project_name, mime)
        self.add_mime_data('sg_status_list', self.sg_status_list, mime)
        self.add_mime_data('shotgrid_id', 'CustomEntity09:%s' % self.id, mime)

        return mime

    def mouseMoveEvent(self, e):

        if e.buttons() == QtCore.Qt.LeftButton:
            drag = QtGui.QDrag(self)
            mime = self.add_entity_mime_data()
            drag.setMimeData(mime)
            pixmap = QtGui.QPixmap(self.size())
            self.render(pixmap)
            #pixmap = pixmap.scaled(100, 32)
            drag.setPixmap(pixmap)
            drag.exec_(QtCore.Qt.MoveAction)

    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.RightButton:
            p = self.mapToGlobal(event.pos())  # or QtGui.QCursor.pos()
            menu = self.create_menu_contextual()
            action = menu.exec_(p)
            if action is not None:
                print(action.text())

    def create_menu_contextual(self):
        menu = QtWidgets.QMenu()
        menu.addAction("Action1")
        menu.addAction("Action2")
        return menu

class CardList(QtWidgets.QWidget):
    clicked = QtCore.Signal(database.View)

    def __init__(self,
                 entity_type,
                 manager,
                 name,
                 items_view=None,
                 drag=False,
                 drop=False,
                 card_type=EntityCard,
                 columns=4):
        super().__init__(parent=manager)

        self.entity_type = entity_type
        self.manager = manager
        self.items_view = items_view
        self.name = name
        self.columns = 4
        self.card_type = card_type
        self.columns = columns
        self.card_widgets = []
        self.allow_drag = drag
        self.allow_drop = drop
        self.setObjectName('no_squared_darker')

        self.setAcceptDrops(True)

        self.main_layout = QtWidgets.QVBoxLayout()

        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        self.main_widget = QtWidgets.QWidget()
        self.main_widget.setObjectName('no_squared_darker')
        self.main_widget.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.scroll_layout = QtWidgets.QGridLayout()
        self.scroll_layout.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignTop)
        self.scroll_layout.setSpacing(10)
        self.scroll_layout.setContentsMargins(10, 10, 10, 10)

        self.main_widget.setLayout(self.scroll_layout)

        self.scroll_area = QtWidgets.QScrollArea(self)
        self.scroll_area.setObjectName('no_squared_darker')
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setWidget(self.main_widget)
        self.scroll_area.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.main_layout.addWidget(self.scroll_area)

        self.setLayout(self.main_layout)

    def show_widgets(self):
        index = 0
        for asset_widget in self.card_widgets:
            if asset_widget.status is not None:
                row, column = self.index_to_cell(index)

                self.scroll_layout.addWidget(asset_widget, row, column)
                index += 1

    def resize_widget(self):
        new_size = self.size()
        width = new_size.width() - 10
        card_width = self.card_type.width + 10
        self.columns = int(width / card_width)
        if self.columns < 1:
            self.columns = 1
        self.show_widgets()

    def index_to_cell(self, index):
        column = index % self.columns
        row = int(index / self.columns)
        return row, column

    def create_card(self, item):
        print(item)
        asset_widget = self.card_type(item,
                                      manager=self.manager,
                                      card_list=self)
        asset_widget.tab = self.name
        self.card_widgets.append(asset_widget)
        return asset_widget


    def clear(self):
        for widget in self.card_widgets:
            widget.deleteLater()
        self.card_widgets = []

    def update_cards(self, items_view):
        self.resize_widget()
        self.clear()
        self.items_view = items_view.get_thumbnails()
        for item in sorted(self.items_view):
            card_widget = self.create_card(item)

        self.show_widgets()

        QtWidgets.QApplication.processEvents()

        for card_widget in self.card_widgets:
            card_widget.update_widgets.start()


    def dragEnterEvent(self, event):
        event.accept()

    def get_drop_entity(self, mime_data):
        shotgrid_id_format = 'application/shotgrid/shotgrid_id'
        full_shotgrid_id = str(mime_data.data(shotgrid_id_format), 'utf-8')
        entity_name, record_id = full_shotgrid_id.split(':', 1)
        record_id = int(record_id)
        item = self.manager.database[entity_name].find_with_filters(id=record_id, single_item=True)
        return item

    def dropEvent(self, event):
        source = event.source()
        if source.tab == self.name:
            return
        mime = event.mimeData()
        item = self.get_drop_entity(mime)
        correct_item = self.manager.process_drop(item, self.entity_type)
        if not correct_item:
            return
        self.create_card(correct_item)
