"""
To use, make sure that externalDropCallback.py is in your MAYA_PLUG_IN_PATH
then do the following:

import maya
maya.cmds.loadPlugin("externalDropCallback.py")

Drag and drop events should appear in the script editor output.
"""
import os
import sys
import ast

import maya.api.OpenMaya as OpenMaya
import maya.OpenMayaUI as OpenMayaUI
import maya.api.OpenMayaUI as OpenMayaUI2
import maya.cmds as cmds
try:
    from shiboken2 import wrapInstance
except:
    from shiboken6 import wrapInstance

from PySide import QtWidgets, QtCore

from pxr import Usd, UsdGeom, Sdf

from pprint import  pprint
# callback
class PyExternalDropCallback(OpenMayaUI.MExternalDropCallback):
    instance = None

    def __init__(self):
        self.asset_variants = {}
        OpenMayaUI.MExternalDropCallback.__init__(self)

    def actionClicked(self, action):
        self.selected_variant = str(action.text())


    def get_point_position(self):
        pos = cmds.autoPlace(um=True)
        x = pos[0]
        y = pos[1]
        z = pos[2]

        pos_point = OpenMaya.MPoint(pos[0], pos[1], pos[2])

        end_pos = OpenMaya.MPoint()
        dir = OpenMaya.MVector()
        view = OpenMayaUI2.M3dView.active3dView()

        output = view.worldToView(pos_point)

        view.viewToWorld(int(output[0]), int(output[1]), end_pos, dir)
        pos2 = OpenMaya.MFloatPoint(end_pos.x, end_pos.y, end_pos.z)
        for mesh in cmds.ls(type='mesh'):
            selectionList = OpenMaya.MSelectionList()
            selectionList.add(mesh)
            dagPath = selectionList.getDagPath(0)
            fnMesh = OpenMaya.MFnMesh(dagPath)
            intersection = fnMesh.closestIntersection(
                OpenMaya.MFloatPoint(pos2),
                OpenMaya.MFloatVector(dir),
                OpenMaya.MSpace.kWorld,
                99999,
                False
            )
            if intersection:
                x = intersection[0].x
                y = intersection[0].y
                z = intersection[0].z

        return (x,y,z), (int(output[0]), int(output[1]))

    def externalDropCallback(self, doDrop, controlName, data):
        if not doDrop:
            return True

        drop_position, position_2d = self.get_point_position()

        urls = data.urls()
        path = ''
        for (i, url) in enumerate(urls):
            if i == 0:
                path = url.replace('file:///', '')

        if not os.path.exists(path):
            return

        drop_data = {}
        if data.hasText():
            try:
                drop_data = ast.literal_eval(data.text())
            except:
               pass
        if not drop_data:
            return

        step = drop_data.get('step', 'Model')
        asset_name = drop_data.get('asset_name', 'drop_item')
        asset_type = drop_data.get('asset_type', 'model')

        if step == 'Rig':
            node_name, instance_number = self.create_rig_assembly(path, asset_name, drop_position, asset_type=drop_data.get('asset_type', ''))
            cmds.setAttr('%s.assetType' % node_name, drop_data.get('asset_type', ''), type='string')

        elif step == 'SetDressing':
            node_name , instance_number = self.create_set_assembly(path, asset_name, position_2d)
            if asset_type != 'Sets':
                cmds.setAttr('%s.translateX' % node_name, drop_position[0])
                cmds.setAttr('%s.translateY' % node_name, drop_position[1])
                cmds.setAttr('%s.translateZ' % node_name, drop_position[2])

        else:
            node_name , instance_number = self.create_model_assembly(path,
                                                                     asset_name,
                                                                     position_2d)

            cmds.setAttr('%s.assetType' % node_name, drop_data.get('asset_type', ''), type='string')
            cmds.setAttr('%s.translateX' % node_name, drop_position[0])
            cmds.setAttr('%s.translateY' % node_name, drop_position[1])
            cmds.setAttr('%s.translateZ' % node_name, drop_position[2])

        cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
        cmds.setAttr('%s.instanceNumber' % node_name, instance_number)

        return True

    def get_root_control(self, root):
        all_ctrls = []
        for node in cmds.listRelatives(root, ad=True, f=True):
            node_type = cmds.nodeType(node)
            if node_type != 'nurbsCurve':
                continue
            parent_node = node.rsplit('|', 1)[0]
            all_ctrls.append(parent_node)

        root_control = sorted(all_ctrls)[0]
        return root_control

    def create_rig_assembly(self, path, asset_name, drop_position, asset_type=''):
        instance_number = self.get_instance_number(asset_name, 'UsdAssembly')
        assembly_name = '%s_%03d' % (asset_name, instance_number)
        asset_type = asset_type.replace(' ', '_')
        node_name = cmds.createNode('RigAssembly', name=assembly_name)

        cmds.setAttr('%s.assetType' % node_name, asset_type, type='string')
        cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
        cmds.setAttr('%s.rigPath' % node_name, path, type='string')
        root_control = self.get_root_control(assembly_name)

        cmds.setAttr('%s.translateX' % root_control, drop_position[0])
        cmds.setAttr('%s.translateY' % root_control, drop_position[1])
        cmds.setAttr('%s.translateZ' % root_control, drop_position[2])

        return node_name, instance_number

    def create_set_assembly(self, path, asset_name, position_2d):
        all_variants = []
        stage = Usd.Stage.Open(path)
        root_prim = stage.GetDefaultPrim()
        prim_sets = root_prim.GetVariantSets()
        if prim_sets.HasVariantSet('setDressing_variant'):
            model_variant_set = prim_sets.GetVariantSet('setDressing_variant')
            all_variants = model_variant_set.GetVariantNames()

        self.selected_variant = None

        if len(all_variants) == 0:
            pass
        elif len(all_variants) == 1:
            self.selected_variant = all_variants[0]
        else:
            view = OpenMayaUI2.M3dView.active3dView()
            model_editor_qwidget = wrapInstance(int(view.widget()), QtWidgets.QWidget)

            height = model_editor_qwidget.height()
            menu = QtWidgets.QMenu(parent=model_editor_qwidget)
            for variant_name in all_variants:
                menu.addAction(variant_name)

            menu.triggered.connect(self.actionClicked)

            qPost = QtCore.QPoint(position_2d[0], height - position_2d[1])
            global_pos = model_editor_qwidget.mapToGlobal(qPost)

            menu.exec_(global_pos)

        instance_number = self.get_instance_number(asset_name, 'SetAssembly')
        assembly_name = '%s_%03d' % (asset_name, instance_number)
        node_name = cmds.createNode('SetAssembly', name=assembly_name)

        cmds.setAttr('%s.setPath' % node_name, path, type='string')
        if self.selected_variant and cmds.attributeQuery('setDressing_variant', n=node_name, exists=True):
            values = cmds.attributeQuery('setDressing_variant', n=node_name, listEnum=True)
            values = values[0].split(':')
            index = values.index(self.selected_variant)
            cmds.setAttr('%s.setDressing_variant' % node_name, index)

        return node_name, instance_number


    def get_instance_number(self, asset_name, assembly_type):
        max_instance = 0
        for previous_asset in cmds.ls(type=assembly_type):
            other_name = cmds.getAttr('%s.assetName' % previous_asset)
            if other_name != asset_name:
                continue
            instance_number = int(cmds.getAttr('%s.instanceNumber' % previous_asset))

            if instance_number > max_instance:
                max_instance = instance_number

        instance_number = max_instance + 1

        return instance_number


    def create_model_assembly(self, path, asset_name, position_2d):
        all_variants = []
        stage = Usd.Stage.Open(path)
        root_prim = stage.GetDefaultPrim()
        prim_sets = root_prim.GetVariantSets()
        if prim_sets.HasVariantSet('model_variant'):
            model_variant_set = prim_sets.GetVariantSet('model_variant')
            all_variants = model_variant_set.GetVariantNames()

        self.selected_variant = None

        if len(all_variants) == 0:
            pass
        elif len(all_variants) == 1:
            self.selected_variant = all_variants[0]
        else:
            view = OpenMayaUI2.M3dView.active3dView()
            model_editor_qwidget = wrapInstance(int(view.widget()), QtWidgets.QWidget)

            height = model_editor_qwidget.height()
            menu = QtWidgets.QMenu(parent=model_editor_qwidget)
            for variant_name in all_variants:
                menu.addAction(variant_name)

            menu.triggered.connect(self.actionClicked)

            qPost = QtCore.QPoint(position_2d[0], height - position_2d[1])
            global_pos = model_editor_qwidget.mapToGlobal(qPost)

            menu.exec_(global_pos)

        instance_number = self.get_instance_number(asset_name, 'UsdAssembly')
        assembly_name = '%s_%03d' % (asset_name, instance_number)
        node_name = cmds.createNode('UsdAssembly', name=assembly_name)

        cmds.setAttr('%s.usdPath' % node_name, path, type='string')

        if self.selected_variant and cmds.attributeQuery('model_variant', n=node_name, exists=True):
            values = cmds.attributeQuery('model_variant', n=node_name, listEnum=True)
            values = values[0].split(':')
            index = values.index(self.selected_variant)
            cmds.setAttr('%s.model_variant' % node_name, index)

        return node_name, instance_number


def initializePlugin(plugin):
    try:
        PyExternalDropCallback.instance = PyExternalDropCallback()
        OpenMayaUI.MExternalDropCallback.addCallback(PyExternalDropCallback.instance)
        sys.stdout.write("Successfully registered callback: PyExternalDropCallback\n")
    except:
        sys.stderr.write("Failed to register callback: PyExternalDropCallback\n")
        raise

def uninitializePlugin(plugin):
    try:
        OpenMayaUI.MExternalDropCallback.removeCallback(PyExternalDropCallback.instance)
        sys.stdout.write("Successfully deregistered callback: PyExternalDropCallback\n")
    except:
        sys.stderr.write("Failed to deregister callback: PyExternalDropCallback\n")
        raise

