import sys
import logging

from pprint import pprint

from PySide import QtWidgets, QtGui, QtCore
import pandas

import packages_io
import library.core.config_manager as config_manager
import library.ui.labeled as labeled

import shotgrid_lib.database as database
import shotgrid_lib.lib.progress_query_widget as progress_query_widget

import asset_library.lib.card_widgets as card_widgets


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)

ENTITY_TAGS = {'Modeling': {'type': 'Geometry', 'file_tag': 'usd'},
               'Rig': {'type': 'Rig', 'file_tag': 'rig'},
               'SetDressing': {'type': 'Set', 'file_tag': 'usd'}
                     }


ASSET_TYPES = {'Characters': {'step': 'Rig', 'asset_types': ['Main Characters', 'Secondary Characters', 'Tertiary Characters']},
               'Creatures':  {'step': 'Rig', 'asset_types': ['Main Creatures', 'Secondary Creatures', 'Tertiary Creatures']},
               'Props':  {'step': 'Rig', 'asset_types': ['Props', 'Setprops']},
               'Elements':  {'step': 'Model', 'asset_types': ['Props', 'Setprops']},
               'Sets':  {'step': 'SetDressing', 'asset_types': ['Sets']},
               'Other':  {'step': 'Rig', 'asset_types': ['Transition']}
               }


class AssetCardWidget(card_widgets.EntityCard):
    changed_info = QtCore.Signal(database.View)

    def add_info(self, main_layout):
        self.pipeline_step = self.card_list.pipeline_step
        self.variant = 'Master'
        self.published_item = None
        self.published_shading = None
        self.status = None

        self.info_layout = QtWidgets.QVBoxLayout()
        self.info_layout.setSpacing(2)
        self.info_layout.setContentsMargins(10, 1, 10, 1)
        self.pipeline_step_label = QtWidgets.QLabel(self.pipeline_step)
        self.asset_name_label = QtWidgets.QLabel(self.entity_view.code)
        self.pipeline_step_label.setObjectName('Small')
        self.asset_name_label.setObjectName('Title')
        self.info_layout.addWidget(self.pipeline_step_label)
        self.info_layout.addWidget(self.asset_name_label)
        self.variant_label = QtWidgets.QLabel(self.variant)
        self.variant_label.setObjectName('Small')
        self.info_layout.addWidget(self.variant_label)
        self.info_layout.addStretch(1)
        main_layout.addLayout(self.info_layout)

    def get_publish(self, step, variant, all_published):

        if all_published.empty:
            return all_published

        step_view = self.database['Step'][step]
        variant_view = self.database['CustomEntity11'][variant]
        published_items = all_published.find_with_filters(sg_variant=variant_view,
                                                          sg_step=step_view,
                                                          sg_status_list='apr',
                                                          single_item=True)
        if published_items.empty:
            published_items = all_published.find_with_filters(sg_variant=variant_view,
                                                              sg_step=step_view,
                                                              single_item=True)
        if published_items.empty and variant != 'Master':
            published_items = self.get_publish(step, 'Master', all_published)

        return published_items


    def draw_status(self):

        all_published_elements = self.entity_view.sg_published_elements
        self.status = None
        if all_published_elements.empty:
            self.setHidden(True)
            self.status_widget.setColor('red')
            return

        shading_item = None
        published_item = self.get_publish(self.pipeline_step, self.variant, all_published_elements)

        if self.pipeline_step != 'Shading':
            shading_item = self.get_publish('Shading', self.variant, all_published_elements)

        if published_item.empty:
            self.setHidden(True)

            self.status_widget.setColor('red')
            return

        self.setHidden(False)
        self.status = published_item.sg_status_list

        if published_item.sg_status_list == 'cmpt':
            self.status_widget.setColor('green')
        elif published_item.sg_status_list == 'ip':
            self.status_widget.setColor('lightblue')
        elif published_item.sg_status_list == 'wtg':
            self.status_widget.setColor('yellow')
        else:
            self.status_widget.setColor('orange')
        self.published_item = published_item
        self.published_shading = shading_item


    def add_entity_mime_data(self):
        mime = QtCore.QMimeData()
        if not self.published_item:
            return mime

        entity = self.published_item

        file_tag = ENTITY_TAGS[self.pipeline_step]['file_tag']
        relative_path = entity.sg_files.get(file_tag)
        asset_type = entity.sg_asset.sg_asset_type.replace(' ', '_')
        project = entity.project.code
        if self.pipeline_step == 'Rig':
            full_path = '%s/%s' % (entity.sg_published_folder, relative_path)
        else:
            full_path = 'V:/%s/publish/usd/assets/%s/%s/asset_assembly.usda' % (project, asset_type, entity.sg_asset_name)

        data = {'asset_name': entity.sg_asset_name,
                'asset_type': entity.sg_asset.sg_asset_type,
                'model_variant': entity.sg_variant_name,
                'model_version': str(entity.sg_version_number),
                'step': self.pipeline_step
                }

        url_paths = [full_path]

        mime.setText(str(data))
        mime.setUrls(url_paths)


        return mime

class AssetFilterWidget(QtWidgets.QWidget):
    selected = QtCore.Signal(list)
    changed_step = QtCore.Signal(str)

    def __init__(self, config, publish_type=None, parent=None):
        super().__init__(parent=parent)
        self.config = config

        self.main_layout = QtWidgets.QVBoxLayout()

        self.setFixedWidth(300)

        pipeline_steps = list(self.config['Steps'].keys())

        if publish_type and publish_type in pipeline_steps:
            self.publish_type = publish_type
        else:
            self.publish_type = pipeline_steps[0]

        self.pipeline_step_widget = labeled.LabeledCombo(self.publish_type, 'Step', values=pipeline_steps, parent=self)
        self.pipeline_step_widget.setFixedHeight(20)
        self.pipeline_step_widget.currentIndexChanged.connect(self.update_step)
        self.filter_tree = QtWidgets.QTreeWidget()
        self.filter_tree.setObjectName('no_squared')
        self.filter_tree.setHeaderLabels(['Asset type'])

        self.filter_tree.itemSelectionChanged.connect(self.set_selection)

        self.main_layout.addWidget(self.pipeline_step_widget)

        self.main_layout.addWidget(self.filter_tree)

        self.setLayout(self.main_layout)
        self.update_filter_tree()

    def update_step(self, new_step):
        self.changed_step.emit(new_step)
        self.update_filter_tree()

    def get_all_selected(self, item):
        this_level_items = []
        count = item.childCount()
        if count == 0:
            return [item.text(0)]
        else:
            for index in range(count):
                child_item = item.child(index)
                children = self.get_all_selected(child_item)
                if children:
                    this_level_items += children
        return this_level_items

    def set_selection(self):
        selected_items = self.filter_tree.selectedItems()
        if selected_items:
            all_children = self.get_all_selected(selected_items[0])
            self.selected.emit(all_children)

    @property
    def pipeline_step(self):
        return self.pipeline_step_widget.getValue()

    def fill_tree(self, tree_data, parent_item=None):
        if isinstance(tree_data, dict):
            for item_name, data in tree_data.items():
                new_item = QtWidgets.QTreeWidgetItem([item_name])

                if parent_item:
                    parent_item.addChild(new_item)
                else:
                    self.filter_tree.addTopLevelItem(new_item)

                self.fill_tree(data, parent_item=new_item)
                new_item.setExpanded(True)

        else:
            for item_name in tree_data:
                new_item = QtWidgets.QTreeWidgetItem([item_name])
                parent_item.addChild(new_item)

    def update_filter_tree(self):
        current_step = self.pipeline_step_widget.getValue()
        selected_items = self.filter_tree.selectedItems()
        old_selection = None
        if selected_items:
            old_selection = selected_items[0].text(0)

        self.filter_tree.clear()
        step_data = self.config['Steps'][current_step]

        self.fill_tree(step_data)

        if old_selection:
            item = self.filter_tree.findItems(old_selection, QtCore.Qt.MatchExactly|QtCore.Qt.MatchRecursive)
            if item:
                self.filter_tree.setItemSelected(item[0], True)

class CardsManager(QtWidgets.QWidget):
    clicked = QtCore.Signal(database.View)

    def __init__(self, project, publish_type=None, parent=None):
        super().__init__(parent=parent)
        self.setAcceptDrops(True)
        self.project = project
        self.publish_type= publish_type
        self.main_module = 'asset_library'
        self.cached_images = {}
        self.config_manager = config_manager.ConfigSolver(project=self.project)
        self.project_config = self.config_manager.project_config
        self.asset_library_config = self.config_manager.get_config(self.main_module, module=self.main_module)

        self.progress_query = progress_query_widget.ProgressQuery(self.project, parent=self)
        self.progress_query.set_precache_entities(['Asset', 'Step', 'CustomEntity11'])
        self.progress_query.precache_dependencies = {'Asset': ['sg_published_elements']}
        self.database = self.progress_query.database
        self.progress_query.finished.connect(self.init_asset_library)

        self.card_widgets = {}
        self.setObjectName('squared_darker')
        self.main_layout = QtWidgets.QHBoxLayout()

        self.main_layout.setSpacing(10)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        self.filter_widget = AssetFilterWidget(self.asset_library_config, parent=self, publish_type=publish_type)
        self.filter_widget.selected.connect(self.init_asset_library)
        self.filter_widget.changed_step.connect(self.clear_card_list)

        self.main_layout.addWidget(self.filter_widget)
        self.widget_tabs = {}

        self.card_list_widget = card_widgets.CardList('Asset',
                                             self,
                                             'source',
                                             drag=True,
                                             drop=False,
                                             card_type=AssetCardWidget)


        self.main_layout.addWidget(self.card_list_widget)

        self.setLayout(self.main_layout)

    def init_asset_library(self, asset_types=[]):
        if not asset_types:
            return
        if 'All' in asset_types:
            self.asset_view = self.database['Asset']
        else:
            self.asset_view = self.database['Asset'].find_with_filters(sg_asset_type=asset_types)

        self.card_list_widget.pipeline_step = self.filter_widget.pipeline_step
        self.card_list_widget.update_cards(self.asset_view)

    def process_drop(self, drop_item, target_entity_type):
        if target_entity_type != 'CustomEntity11' or drop_item.type != 'Asset':
            return

        variant_view = self.database['CustomEntity11']['Master']
        breakdown_data = {'sg_asset': drop_item.as_shotgun(),
                          'sg_link': self.shot_view.as_shotgun(),
                          'project': self.shot_view.project.as_shotgun(),
                          'sg_instance': 1,
                          'sg_alias': '',
                          'sg_geometry_variant': variant_view.as_shotgun(),
                          'sg_shading_variant': variant_view.as_shotgun(),
                          'sg_status_list': 'wtg',
                          'image': drop_item.image,
                          'type': 'CustomEntity11'
                          }
        serie = pandas.DataFrame([breakdown_data])
        view = database.View(serie, 'CustomEntity12', unique=True)
        return view

    def clear_card_list(self):
        self.card_list_widget.clear()

    def start(self):
        self.progress_query.start()

    def resize_widgets(self, new_size):
        self.card_list_widget.resize_widget()

class AssetCardShowWindow(packages_io.MainWindow):

    def __init__(self, project, publish_type=None):
        super(AssetCardShowWindow, self).__init__('Asset Library', style_sheet='named_style')
        self.project = project
        self.publish_type = publish_type

        self.setContentsMargins(0, 0, 0, 0)
        self.setMinimumHeight(500)
        self.setMinimumWidth(900)
        self.setWindowTitle('Asset Library')
        self.apply_style(self)

        self.main_widget = CardsManager(self.project, publish_type=publish_type)

        self.setCentralWidget(self.main_widget)

        QtWidgets.QApplication.processEvents()
        self.main_widget.start()

    def resizeEvent(self, event):
        self.main_widget.resize_widgets(event.size())

def open_asset_library(project=None, publish_type='None'):

    window_test = AssetCardShowWindow(project=project, publish_type=publish_type)
    window_test.show()

if __name__ == '__main__':
    app = QtWidgets.QApplication(sys.argv)

    window_test = AssetCardShowWindow(project='tpt', publish_type='Model')
    window_test.show()

    app.exec_()
