import os
import logging
import importlib

import maya.cmds as cmds
from pxr import UsdShade


import usd.lib.usd_manager as usd_manager
import actions.maya.shading.create_preview_shader as create_preview_shaders

importlib.reload(usd_manager)
importlib.reload(create_preview_shaders)

logger = logging.getLogger(__name__)


def create_shader_placeholder(asset_name,
                              asset_type,
                              output_path,
                              project):

    manager = usd_manager.UsdManager(project)
    manager.set_entity(asset_name, 'Asset', asset_type=asset_type)
    manager.open(output_path)
    manager.save_stage()

    return {'usd_path': output_path}

def export_usd_shaders(asset_root,
                       asset_name,
                       asset_type,
                       output_path,
                       shading_variant,
                       shading_version,
                       shading_shotgrid_id,
                       project,
                       maya_path=None
                       ):
    logging.debug('Export USD Shaders')
    logging.debug('Output_path: %s' % output_path)
    logging.debug('Asset_type: %s' % asset_type)
    logging.debug('Asset_name: %s' % asset_name)
    logging.debug('Variant: %s' % shading_variant)
    logging.debug('Version: %s' % shading_version)
    cmds.loadPlugin('mayaUsdPlugin.py')

    if not cmds.objExists(asset_root):
        return

    cmds.select(asset_root, r=True)

    dirname = os.path.dirname(output_path)
    if not os.path.exists(dirname):
        os.makedirs(dirname)

    #cmds.arnoldExportAss(f=output_path, mask=24, s=True, exportAllShadingGroups=True)
    cmds.setAttr('|asset|proxy.visibility', 1)
    cmds.arnoldExportAss(f=output_path, mask=24, s=True, exportAllShadingGroups=True)

    cmds.setAttr('|asset|proxy.visibility', 0)
    manager = usd_manager.UsdManager(project)
    manager.set_entity(asset_name, 'Asset', asset_type=asset_type)
    manager.open(output_path)
    manager.stage.DefinePrim('/asset')
    manager.repath_prim('/render', '/asset/render')
    manager.repath_prim('/proxy', '/asset/proxy')

    manager.save_stage()

    for prim in manager.stage.Traverse():
        if prim.GetTypeName() != 'Mesh':
            continue
        value = manager.get_attribute(prim, 'primvars:arnold:subdiv_type')
        if value is None:
            manager.create_attribute(prim, 'primvars:arnold:subdiv_type', attribute_type='token', value='none')
            manager.set_attribute(prim, 'primvars:arnold:subdiv_iterations', 0)

    manager.remove_attributes('shading', flatten_meshes=True)

    mtl_prim = manager.stage.DefinePrim('/asset/mtl')
    manager.set_prim_typename('/asset/mtl', 'Scope')


    to_translate = {}
    for prim in manager.stage.Traverse():
        if prim.GetTypeName() != 'Material':
            continue

        prim_name = prim.GetName()
        new_path = '/asset/mtl/%s' % prim_name
        prim_path = str(prim.GetPath())
        to_translate[prim_path] = new_path
    print('-' * 100)
    for prim_path, new_path in to_translate.items():
        print('translate', prim_path, new_path)
        manager.repath_prim(prim_path, new_path)


    create_preview_shaders.create_preview_from(manager.stage)

    print('create spheres')

    for prim in manager.stage.Traverse():
        if prim.GetTypeName() != 'Material':
            continue
        prim_name = prim.GetName()
        print('create sphere %s' % prim_name)
        material = UsdShade.Material.Define(manager.stage, prim.GetPath())

        sphere_path = '/asset/guides/sph_%s' % prim_name
        sphere_prim = manager.stage.DefinePrim(sphere_path, 'Sphere')
        sphere_prim.ApplyAPI(UsdShade.MaterialBindingAPI)


        manager.set_prim_purpose(sphere_path, 'guide')
        binding = UsdShade.MaterialBindingAPI(sphere_prim)
        binding.Bind(material)

    xform_prims = []
    for prim in manager.stage.Traverse():
        if prim.GetTypeName() in ['Xform', 'Mesh']:
            xform_prims.append(prim)

    #for prim in xform_prims:
    #    manager.override_prim(prim.GetPath())

    asset_prim = manager.stage.GetPrimAtPath('/asset')

    manager.create_attribute(asset_prim, 'atlantis:asset_name', value=asset_name)
    manager.create_attribute(asset_prim, 'atlantis:asset_type', value=asset_type)
    manager.create_attribute(asset_prim, 'atlantis:shading_variant', value=shading_variant)
    manager.create_attribute(asset_prim, 'atlantis:shading_version', value=shading_version, attribute_type=int)
    manager.create_attribute(asset_prim, 'atlantis:shading_shotgrid_id', value=shading_shotgrid_id, attribute_type=int)
    manager.create_attribute(asset_prim, 'atlantis:shading_maya_path', value=maya_path, attribute_type='path')

    manager.create_attribute(asset_prim, 'atlantis:path', value=output_path, attribute_type='path')

    #new_path = output_path.replace('.usda', '_changed.usda')
    #manager.save_as(new_path)
    manager.save_stage()
    print(output_path)
    return {'usd': output_path}

if __name__ == "__main__":
    create_shader_placeholder()

# main()