import json
import os
import logging
import maya.cmds as cmds

logger = logging.getLogger(__name__)


def get_set_data(node):
    node_data = {}
    node_data['name'] = node
    node_type = cmds.nodeType(node)

    node_data['translate'] = cmds.getAttr('%s.translate' % node)[0]
    node_data['rotate'] = cmds.getAttr('%s.rotate' % node)[0]
    node_data['scale'] = cmds.getAttr('%s.scale' % node)[0]
    node_data['type'] = node_type

    if node_type == 'Assembly':
        logger.info('Export Assembly %s' % node)
        assembly_type = cmds.getAttr('%s.assemblyType' % node)
        node_data['assemblyType'] = assembly_type
        path = cmds.getAttr('%s.geometryPath' % node)
        shader_path = cmds.getAttr('%s.shadingPath' % node)
        representation = cmds.getAttr('%s.activeRepresentation' % node)
        node_data['path'] = path
        node_data['representation'] = representation
        node_data['shader_path'] = shader_path

        logger.debug('  Path: %s' % node_data['path'])
        logger.debug('  assemblyType: %s' % node_data['assemblyType'])
        logger.debug('  representation: %s' % node_data['representation'])
        logger.debug('  shader_path: %s' % node_data['shader_path'])

    elif node_type == 'transform':
        logger.debug('Export transform %s' % node)

        children = cmds.listRelatives(node, c=True)
        node_data['children'] = []
        for child in children:
            child_data = get_set_data(child)
            node_data['children'].append(child_data)

    return node_data


def publish_set(root='', output=''):
    json_folder = os.path.dirname(output)

    if not os.path.exists(json_folder):
        os.makedirs(json_folder)

    logger.info('Running publish json set')
    logger.debug('Root: %s' % root)
    set_data = get_set_data(root)

    with open(output, 'w') as f:
        json.dump(set_data, f, ensure_ascii=False, indent=4)

    logger.info('Done')
    return {'root': root,
            'json_path': output}

if __name__ == '__main__':
    import argparse

    logger.info('Running export alembic')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-s', '-source', '--source-file', dest='source', help='Source file path')
    parser.add_argument('-o', '-output', '--output-file', dest='output', help='Output file path')
    parser.add_argument('-r', '-root', '--root-geo', dest='root', help='Root geometry')

    args = parser.parse_args()

    import maya.standalone as standalone

    standalone.initialize(name='python')

    cmds.loadPlugin('AbcImport')

    cmds.file(args.source, o=True)


    publish_set(args.root, args.output)