import os

import importlib
from pprint import pprint
import importlib.resources as imp_resources
import library.core.config_manager as config_manager
import shotgrid_lib.database as database

def set_render_options(manager, shot_name, images_path, start_frame, end_frame, project):

    sg_database = database.DataBase()
    sg_database.fill(project=project, precatch=False)
    shot_filters = [['code', 'is', shot_name]]
    sg_database.query_sg_database('Shot', filters=shot_filters)
    shot_view = sg_database['Shot'][shot_name]
    shot_resolution = shot_view.sg_resolution
    if not shot_resolution:
        shot_resolution = 'Default'

    config_solver = config_manager.ConfigSolver(project=project)
    resolutions_data = config_solver.get_config('show_resolutions')

    shot_data = resolutions_data.get(shot_resolution)


    stage = manager.stage
    options_prim = stage.GetPrimAtPath('/options')
    attr = options_prim.GetAttribute('arnold:camera')
    attr.Set('/Cameras/camera')
    manager.create_attribute(options_prim, 'arnold:xres', value=shot_data['width'], attribute_type=int)
    manager.create_attribute(options_prim, 'arnold:yres', value=shot_data['height'], attribute_type=int)

    drivers_prim = stage.GetPrimAtPath('/defaultArnoldDriver')

    for child in drivers_prim.GetChildren():
        attr = child.GetAttribute('arnold:filename')
        filename = attr.Get()

        name_attr = child.GetAttribute('arnold:name')
        aov_name = name_attr.Get().split('.')[-1]
        if aov_name == 'RGBA':
            aov_name = 'beauty'

        aov_folder = '%s/%s' % (images_path, aov_name)
        if not os.path.exists(aov_folder):
            os.makedirs(aov_folder)

        for frame in range(start_frame, end_frame + 1):
            frame_str = '%04d' % frame
            new_path = '%s/%s.%s.exr' % (aov_folder, shot_name, frame_str)
            # new_path = Sdf.AssetPath(new_path)
            attr.Set(time=frame, value=new_path)


    ocio_path = os.environ.get('OCIO', 'V:/company/tools/ocio/0.1.0/config/config.ocio')

    for prim in stage.Traverse():
        if prim.GetTypeName() != 'ArnoldColorManagerOcio':
            continue

        ocio_attr = prim.GetAttribute('arnold:config')
        ocio_attr.Set(value=ocio_path)

def create_postanim(project,
                    output_path,
                    shot_name,
                    usd_cache,
                    start_frame,
                    end_frame):

    start_frame = int(start_frame)
    end_frame = int(end_frame)

    print('create post anim')

    import usd.lib.usd_manager as usd_manager
    importlib.reload(usd_manager)

    manager = usd_manager.UsdManager(project)
    manager.set_entity(shot_name, 'Shot')
    path = manager.filename
    manager.open(path)
    root_folder = os.path.dirname(output_path)


    if os.path.exists(output_path):
        os.remove(output_path)
    new_shot = usd_manager.UsdManager(project)
    new_shot.set_entity(shot_name, 'Shot')
    new_shot.open(output_path)

    options_path = imp_resources.path('actions.resources', 'post_anim_settings.usda' )
    options_path = str(options_path).replace('\\', '/')

    new_shot.stage.SetStartTimeCode(start_frame)
    new_shot.stage.SetEndTimeCode(end_frame)


    print(options_path)
    new_shot.add_sublayer(str(options_path))
    new_shot.add_sublayer(str(usd_cache))


    all_layers = manager.get_layers()
    pprint(all_layers)
    mute_layers = ['assembly','animation', 'postanim', ]
    for layer_name, layer_path in all_layers.items():
        if layer_name in mute_layers:
            continue
        new_shot.add_sublayer(str(layer_path.identifier))

    images_path = root_folder

    new_shot.print()

    set_render_options(new_shot, shot_name, images_path, start_frame, end_frame, project)
    new_shot.print()
    new_shot.save_stage()
    print('saved: %s' % output_path)




if __name__ == '__main__':
    import argparse
    import sys

    #sys.path.insert(0, 'V:/company/tools/pixar_usd')

    print('Running create post anim')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-p', '-project', '--project', dest='project', help='Project code')
    parser.add_argument('-s', '-shot', '--shot-name', dest='shot_name', help='Shot code')
    parser.add_argument('-op', '-output', '--output-path', dest='output_path', help='Output file folder')
    parser.add_argument('-c', '-cache', '--cache-path', dest='cache_path', help='Cache path')
    parser.add_argument('-sf', '-start', '--start-frame', dest='start_frame', help='First frame to cache')
    parser.add_argument('-ef', '-end', '--end-frame', dest='end_frame', help='Last frame to cache')

    args = parser.parse_args()

    create_postanim(args.project,
                    args.output_path,
                    args.shot_name,
                    args.cache_path,
                    args.start_frame,
                    args.end_frame)