import logging
logger = logging.getLogger(__name__)

import importlib

import library.core.config_manager as config_manager

import maya.cmds as cmds
import usd.lib.usd_manager as usd_manager
from actions.maya.general import maya_sets_usd_collections as maya_inputs

importlib.reload(usd_manager)



def load_prim_at_node(node_name, filename, prim_vars):

    node_path = cmds.ls(node_name, long=True)[0]
    logger.debug(f'{node_name} geometry root: {node_path}')
    render_prim_path = node_path.replace('|', '/')  # '/asset/render'
    parent = node_path.rsplit('|', 1)[0]
    logger.debug(f'{node_name} geometry parent: {parent}')

    logger.info('Importing')

    if cmds.objExists(node_path):
        cmds.delete(node_path)

    cmds.mayaUSDImport(file=filename,
                       primVariant=prim_vars,
                       primPath=render_prim_path,
                       parent=parent)
    return parent

def load_usd_model(step_data=None, project='', current_step=''):
    logger.info('== Running Load usd model ==')

    pipeline_step = current_step if current_step else step_data['step']

    logger.debug('Asset name: %s' % pipeline_step)

    asset_name = step_data['code']
    logger.debug('Asset name: %s' % asset_name)

    asset_type = step_data['asset_type']
    logger.debug('Asset type: %s' % asset_type)

    variant = step_data['variant']
    logger.debug('Asset variant: %s' % variant)

    version_number = step_data['version_number']
    logger.debug('Asset version number: %s' % version_number)

    entity_type='Asset'
    manager = usd_manager.UsdManager(project)
    manager.set_entity(asset_name, entity_type, asset_type=asset_type)
    filename = manager.get_layer_filename('geometry')
    manager.open(filename)

    render_prim = manager.stage.GetPrimAtPath('/asset/render')
    proxy_prim = manager.stage.GetPrimAtPath('/asset/proxy')
    guide_prim = manager.stage.GetPrimAtPath('/asset/guide')

    logger.info(f'Loading usd geomtry from {filename}')

    prim_vars = [['/asset', 'model_variant', variant],
                 ['/asset', 'model_version', '%03d' % version_number]]

    config_solver = config_manager.ConfigSolver(project=project)
    templates_data = config_solver.get_config('scene_template', module='builder')

    render_geo = templates_data[pipeline_step].get('render_geo')
    proxy_geo = templates_data[pipeline_step].get('proxy_geo')

    guides_group = templates_data[pipeline_step].get('guides')

    render_geo = cmds.ls(render_geo, long=True)[0]

    logger.debug('Render geometry root: %s' % render_geo)

    parent = load_prim_at_node(render_geo, filename, prim_vars)

    if cmds.objExists(proxy_geo) and proxy_prim.IsValid():
        load_prim_at_node(proxy_geo, filename, prim_vars)

    if guide_prim.IsValid():

        load_prim_at_node(guides_group, filename, prim_vars)

    logger.info('Adding asset attributes')

    cmds.addAttr(parent, longName='asset_name', dataType='string')
    cmds.setAttr('%s.asset_name' % parent, step_data['code'], type='string')

    cmds.addAttr(parent, longName='asset_type', dataType='string')
    cmds.setAttr('%s.asset_type' % parent, step_data['asset_type'], type='string')

    cmds.addAttr(parent, longName='model_variant', dataType='string')
    cmds.setAttr('%s.model_variant' % parent, step_data['variant'], type='string')

    cmds.addAttr(parent, longName='model_version', dataType='string')
    cmds.setAttr('%s.model_version' % parent, step_data['version_number'], type='string')

    maya_inputs.usd_collection_to_maya_set(manager)