import logging

import maya.cmds as cmds
import mayaUsd

logger = logging.getLogger(__name__)


ASSET_NODES={'Cameras': {'suffix': 'CAM', 'group':'Camera'},
             'Main Characters': {'suffix': 'CHAR', 'group':'Characters'},
             'Secondary Characters': {'suffix': 'CHAR', 'group': 'Characters'},
             'Setprops': {'suffix': 'PROP', 'group': 'Props'},
             'Sets': {'suffix': 'SET', 'group':'Sets'},
        }

def load_usd_layer(step_data=None, file_tag='usd', asset_name=None, as_root=False):
    shape = cmds.ls(type="mayaUsdProxyShape", long=True)
    if shape:
        stage_shape = shape[0]
    else:
        stage_shape = cmds.createNode('mayaUsdProxyShape', n='Asset')
        stage_shape = cmds.ls(stage_shape, long=True)[0]  # make sure to use long full paths
        cmds.expression(o=stage_shape, s='%s.time = frame' % stage_shape)

    stage = mayaUsd.ufe.getStage(stage_shape)
    root_layer = stage.GetRootLayer()

    asset_folder = step_data['published_folder']
    asset_file = step_data['files'].get(file_tag)

    source_layer = '%s/%s' % (asset_folder, asset_file)

    cmds.mayaUsdLayerEditor(root_layer.identifier, edit=True, insertSubPath=(0, source_layer))
    return stage_shape


def create_aov(node_name, aov_name, light_group):
    new_aov = cmds.createNode('aiAOV', name=node_name)
    cmds.setAttr('%s.name' % new_aov,aov_name, type='string')
    if light_group:
        lpe = "C.*[<L.'%s'>O]" % light_group

        cmds.setAttr('%s.lightPathExpression' % new_aov, lpe, type='string')
    cmds.connectAttr('defaultArnoldFilter.message', '%s.outputs[0].filter' % new_aov, f=True)
    cmds.connectAttr('defaultArnoldDriver.message', '%s.outputs[0].driver' % new_aov,  f=True)
    cmds.connectAttr('%s.message' % new_aov, 'defaultArnoldRenderOptions.aovList', na=True, f=True)

    return new_aov


def load_light_layer(step_data=None, file_tag='usd', asset_name=None, as_root=False):

    stage_shape = load_usd_layer(step_data=step_data, file_tag=file_tag, asset_name=asset_name, as_root=as_root)
    stage = mayaUsd.ufe.getStage(stage_shape)

    if not cmds.objExists('defaultArnoldRenderOptions'):
        cmds.createNode('aiOptions', n='defaultArnoldRenderOptions')
    if not cmds.objExists('defaultArnoldFilter'):
        cmds.createNode('aiAOVFilter', n='defaultArnoldFilter')
    if not cmds.objExists('defaultArnoldDriver'):
        cmds.createNode('aiAOVDriver', n='defaultArnoldDriver')

    rgba_aov = create_aov('Beauty', 'RGBA', None)

    for prim in stage.Traverse():
        aov_attr = prim.GetAttribute('primvars:arnold:aov')
        if not aov_attr.IsValid():
            continue
        light_group = aov_attr.Get()
        if not light_group:
            continue
        new_aov = create_aov('aiAOV_%s' % light_group, 'test_%s'% light_group, light_group)





def load_usd_assembly(step_data=None, file_tag='usd', asset_name=None, sufix='AST', full_asset=True):
    from pprint import pprint
    logger.debug('load_assembly')
    logger.debug('file_tag: %s' % file_tag)
    logger.debug('asset_name: %s' % asset_name)
    pprint(step_data)
    if not step_data:
        return


    if not cmds.pluginInfo('usd_assembly.py', q=True, l=True):
        cmds.loadPlugin('usd_assembly.py')

    if asset_name is None:
        asset_name = step_data['code']

    asset_type = step_data['asset_type']
    asset_folder = step_data['published_folder']
    asset_file = step_data['files'].get(file_tag)

    if not asset_file:
        return
    if full_asset:
        folder_asset_type = asset_type.replace(' ', '_')
        project = step_data['project']

        full_asset_path = 'V:/%s/publish/usd/assets/%s/%s/model/%s_model.usda' % (project, folder_asset_type, asset_name, asset_name)
    else:
        full_asset_path = '%s/%s' % (asset_folder, asset_file)

    node_name = cmds.createNode('UsdAssembly', name=asset_name)
    cmds.setAttr('%s.usdPath' % node_name, full_asset_path, type='string')
    cmds.setAttr('%s.usdPath' % node_name, lock=1)
    cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
    cmds.setAttr('%s.assetName' % node_name, lock=1)

    cmds.setAttr('%s.assetType' % node_name, asset_type, type='string')
    cmds.setAttr('%s.assetType' % node_name, lock=1)

def load_usd_camera(step_data=None, file_tag='usd', asset_name=None, sufix='AST'):
    logger.debug('load_assembly')
    logger.debug('file_tag: %s' % file_tag)
    logger.debug('asset_name: %s' % asset_name)
    if not step_data:
        return