import logging
from pprint import pprint

import maya.cmds as cmds

logger = logging.getLogger(__name__)

ASSET_NODES={'Others': {'suffix': 'SHOT', 'group': 'Shot_elements'},
             'Main Characters': {'suffix': 'CHAR', 'group':'Characters'},
             'Secondary Characters': {'suffix': 'CHAR', 'group': 'Characters'},
             'Setprops': {'suffix': 'PROP', 'group': 'Props'},
             'Props': {'suffix': 'PROP', 'group': 'Props'},
             'Sets': {'suffix': 'SET', 'group':'Sets'},
             'Transition': {'suffix': 'TPT', 'group': 'Template'},
             'LightRig': {'suffix': 'LGHT', 'group': 'LightRig'},

        }

ASSEMBLY_CONFIG = {'Shader': {'suffix': 'SHAD', 'group':'Shaders'},}



def load_assembly_shaders(step_data=None, asset_name=None, as_root=False):
    logger.debug('load_assembly_shaders test')
    logger.debug('asset_name:%s' % asset_name)

    if not step_data:
        return

    if not cmds.pluginInfo('assembly.py', q=True, l=True):
        cmds.loadPlugin('assembly.py')

    if asset_name is None:
        asset_name = step_data['code']

    asset_folder = step_data['published_folder']
    asset_file = step_data['files'].get('shaders')

    if not asset_file:
        return
    full_asset_path = '%s/%s' % (asset_folder, asset_file)

    assembly_name, group, asset_type = get_assembly_name(step_data, asset_name, as_root)

    if not cmds.objExists(assembly_name):
        node_name = cmds.createNode('RigAssembly', name=assembly_name, parent=group)
    else:
        node_name = assembly_name

    cmds.setAttr('%s.shadingVariant' % node_name, step_data.get('variant', ''), type='string')
    cmds.setAttr('%s.shadingVersion' % node_name, str(step_data.get('version_number', '')), type='string')
    cmds.setAttr('%s.shadingPath' % node_name, full_asset_path, type='string')



def get_assembly_name(step_data, asset_name, as_root):
    alias = step_data.get('alias')
    asset_type = step_data.get('asset_type')
    instance = step_data.get('instance', 0)
    default = {'suffix': 'SHOT', 'group': 'Shot_elements'}

    sufix = ASSET_NODES.get(asset_type, default)['suffix']
    group = ASSET_NODES.get(asset_type, default)['group']

    if alias:
        assembly_name = '%s_%s' % (alias, sufix)
    else:
        assembly_name = '%s_%03d_%s' % (asset_name, instance, sufix)

    if as_root:
        group = None
    else:
        if not cmds.objExists(group):
            group = cmds.createNode('transform', n=group)

    return assembly_name, group, asset_type


def get_publish_path(step_data, file_tag):
    asset_folder = step_data['published_folder']
    asset_file = step_data['files'].get(file_tag)

    if not asset_file:
        return
    full_asset_path = '%s/%s' % (asset_folder, asset_file)

    return full_asset_path


def get_level_of_detail(step_data):
    lod_list = ['Very long', 'Long', 'Medium', 'Close']

    lod = 1
    if 'shot_type' in step_data:
        shot_type = step_data['shot_type']
        if not shot_type:
            shot_type = 'Medium'
        lod = lod_list.index(shot_type)
    return lod


def add_rig_assembly(step_data=None, file_tag='rig', asset_name=None, as_root=False):
    if not cmds.pluginInfo('rig_assembly.py', q=True, l=True):
        cmds.loadPlugin('rig_assembly.py')

    if asset_name is None:
        asset_name = step_data['code']

    full_asset_path = get_publish_path(step_data, file_tag)
    assembly_name, group, asset_type = get_assembly_name(step_data, asset_name, as_root)
    lod = get_level_of_detail(step_data)

    logger.info('Creating rig assembly for asset %s' % assembly_name)


    if not cmds.objExists(assembly_name):
        node_name = cmds.createNode('RigAssembly', name=assembly_name, parent=group)
    else:
        node_name = assembly_name


    cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
    cmds.setAttr('%s.assetName' % node_name, lock=1)
    cmds.setAttr('%s.assetType' % node_name, asset_type, type='string')
    cmds.setAttr('%s.assetType' % node_name, lock=1)
    cmds.setAttr('%s.instanceNumber' % node_name, int(step_data.get('instance', '0')))
    cmds.setAttr('%s.instanceNumber' % node_name, lock=1)

    cmds.setAttr('%s.rigPath' % node_name, full_asset_path, type='string')
    cmds.setAttr('%s.rigPath' % node_name, lock=1)


    cmds.setAttr('%s.rigVariant' % node_name, step_data.get('variant', ''), type='string')
    cmds.setAttr('%s.rigVariant' % node_name, lock=1)
    cmds.setAttr('%s.rigVersion' % node_name, str(step_data.get('version_number', '')), type='string')
    cmds.setAttr('%s.rigVersion' % node_name, lock=1)
    cmds.setAttr('%s.rigHash' % node_name, step_data.get('hash', ''), type='string')
    cmds.setAttr('%s.rigHash' % node_name, lock=1)

def add_geometry_assembly(step_data=None, file_tag='abc_high', asset_name=None, as_root=False):
    if not cmds.pluginInfo('geometry_assembly.py', q=True, l=True):
        cmds.loadPlugin('geometry_assembly.py')

    print('AS ROOT', as_root)
    if asset_name is None:
        asset_name = step_data['code']

    full_asset_path = get_publish_path(step_data, file_tag)
    assembly_name, group, asset_type = get_assembly_name(step_data, asset_name, as_root)
    lod = get_level_of_detail(step_data)

    logger.info('Creating geomtry assembly for asset %s' % assembly_name)

    node_name = cmds.createNode('GeometryAssembly', name=assembly_name, parent=group)
    cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
    cmds.setAttr('%s.assetName' % node_name, lock=1)
    cmds.setAttr('%s.assetType' % node_name, asset_type, type='string')
    cmds.setAttr('%s.assetType' % node_name, lock=1)
    cmds.setAttr('%s.instanceNumber' % node_name, int(step_data.get('instance', '0')))
    cmds.setAttr('%s.instanceNumber' % node_name, lock=1)
    cmds.setAttr('%s.LevelOfDetail' % node_name, lod)
    cmds.setAttr('%s.LevelOfDetail' % node_name, lock=1)

    cmds.setAttr('%s.geometryPath' % node_name, full_asset_path, type='string')
    cmds.setAttr('%s.geometryPath' % node_name, lock=1)
    cmds.setAttr('%s.geometryFileTag' % node_name, file_tag, type='string')
    cmds.setAttr('%s.geometryFileTag' % node_name, lock=1)
    cmds.setAttr('%s.geometryStep' % node_name, step_data['step'], type='string')
    cmds.setAttr('%s.geometryStep' % node_name, lock=1)
    cmds.setAttr('%s.geometryVariant' % node_name, step_data.get('variant', ''), type='string')
    cmds.setAttr('%s.geometryVariant' % node_name, lock=1)
    cmds.setAttr('%s.geometryVersion' % node_name, str(step_data.get('version_number', '')), type='string')
    cmds.setAttr('%s.geometryVersion' % node_name, lock=1)
    cmds.setAttr('%s.geometrySgId' % node_name, step_data.get('id', ''), type='string')
    cmds.setAttr('%s.geometrySgId' % node_name, lock=1)
    cmds.setAttr('%s.geometryHash' % node_name, step_data.get('hash', ''), type='string')
    cmds.setAttr('%s.geometryHash' % node_name, lock=1)

def add_set_assembly(step_data=None, file_tag='abc_high', asset_name=None, as_root=False):
    if not cmds.pluginInfo('set_assembly.py', q=True, l=True):
        cmds.loadPlugin('set_assembly.py')

    if asset_name is None:
        asset_name = step_data['code']

    full_asset_path = get_publish_path(step_data, file_tag)
    assembly_name, group, asset_type = get_assembly_name(step_data, asset_name, as_root)
    lod = get_level_of_detail(step_data)

    logger.info('Creating set assembly for asset %s' % assembly_name)

    node_name = cmds.createNode('SetAssembly', name=assembly_name, parent=group)
    cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
    cmds.setAttr('%s.assetName' % node_name, lock=1)
    cmds.setAttr('%s.instanceNumber' % node_name, int(step_data.get('instance', '0')))
    cmds.setAttr('%s.instanceNumber' % node_name, lock=1)

    cmds.setAttr('%s.setPath' % node_name, full_asset_path, type='string')
    cmds.setAttr('%s.setPath' % node_name, lock=1)
    cmds.setAttr('%s.setFileTag' % node_name, file_tag, type='string')
    cmds.setAttr('%s.setFileTag' % node_name, lock=1)
    cmds.setAttr('%s.setStep' % node_name, step_data['step'], type='string')
    cmds.setAttr('%s.setStep' % node_name, lock=1)
    cmds.setAttr('%s.setVariant' % node_name, step_data.get('variant', ''), type='string')
    cmds.setAttr('%s.setVariant' % node_name, lock=1)
    cmds.setAttr('%s.setVersion' % node_name, str(step_data.get('version_number', '')), type='string')
    cmds.setAttr('%s.setVersion' % node_name, lock=1)
    cmds.setAttr('%s.setSgId' % node_name, step_data.get('id', ''), type='string')
    cmds.setAttr('%s.setSgId' % node_name, lock=1)
    cmds.setAttr('%s.setHash' % node_name, step_data.get('hash', ''), type='string')
    cmds.setAttr('%s.setHash' % node_name, lock=1)

    if step_data.get('children'):
        hide_asset_in_parent(assembly_name, step_data['children'])

def load_assembly_model(step_data=None, assembly_type='Geometry', file_tag='abc_high', asset_name=None, as_root=False):
    logger.debug('load_assembly')
    logger.debug('assembly_type: %s' % assembly_type)
    logger.debug('file_tag: %s' % file_tag)
    logger.debug('asset_name: %s' % asset_name)
    logger.debug('as_root: %s' % as_root)
    logger.debug('shot type: %s' % step_data.get('shot_type', 'asset'))
    lod = get_level_of_detail(step_data)
    pprint(step_data)
    if not step_data:
        return

    if assembly_type == 'LightRig':
        step_data['asset_type'] = 'LightRig'

    if not cmds.pluginInfo('assembly.py', q=True, l=True):
        cmds.loadPlugin('assembly.py')

    if asset_name is None:
        asset_name = step_data['code']

    asset_folder = step_data['published_folder']
    asset_file = step_data['files'].get(file_tag)

    if not asset_file:
        return
    full_asset_path = '%s/%s' % (asset_folder, asset_file)
    alias = step_data.get('alias')

    asset_type = step_data.get('asset_type', 'Shot')
    instance = step_data.get('instance', 0)
    default = {'suffix': 'SHOT', 'group': 'Shot_elements'}

    if assembly_type in ASSEMBLY_CONFIG:
        sufix = ASSEMBLY_CONFIG[assembly_type]['suffix']
        group = ASSEMBLY_CONFIG[assembly_type]['group']

    else:
        sufix = ASSET_NODES.get(asset_type, default)['suffix']
        group = ASSET_NODES.get(asset_type, default)['group']

    if alias:
        assembly_name = '%s_%s' % (alias, sufix)
    else:
        assembly_name = '%s_%03d_%s' % (asset_name, instance, sufix)

    if as_root:
        group = None
    else:
        if not cmds.objExists(group):
            group = cmds.createNode('transform', n=group)

    logger.info('loading assembly of type %s for asset %s' % (assembly_type, assembly_name))
    node_name = cmds.createNode('Assembly', name=assembly_name, parent=group)
    cmds.setAttr('%s.assemblyType' % node_name, assembly_type, type='string')
    cmds.setAttr('%s.assemblyType' % node_name, lock=1)
    cmds.setAttr('%s.path' % node_name, full_asset_path, type='string')
    cmds.setAttr('%s.path' % node_name, lock=1)
    cmds.setAttr('%s.assetName' % node_name, asset_name, type='string')
    cmds.setAttr('%s.assetName' % node_name, lock=1)
    cmds.setAttr('%s.assetType' % node_name, asset_type, type='string')
    cmds.setAttr('%s.assetType' % node_name, lock=1)
    cmds.setAttr('%s.fileTag' % node_name, file_tag, type='string')
    cmds.setAttr('%s.fileTag' % node_name, lock=1)
    cmds.setAttr('%s.pipelineStep' % node_name, step_data['step'], type='string')
    cmds.setAttr('%s.pipelineStep' % node_name, lock=1)
    cmds.setAttr('%s.variant' % node_name, step_data.get('variant', ''), type='string')
    cmds.setAttr('%s.variant' % node_name, lock=1)
    cmds.setAttr('%s.version' % node_name, str(step_data.get('version_number', '')), type='string')
    cmds.setAttr('%s.version' % node_name, lock=1)
    cmds.setAttr('%s.shotgrid_id' % node_name, step_data.get('id', ''), type='string')
    cmds.setAttr('%s.shotgrid_id' % node_name, lock=1)
    cmds.setAttr('%s.hash' % node_name, step_data.get('hash', ''), type='string')
    cmds.setAttr('%s.hash' % node_name, lock=1)
    cmds.setAttr('%s.instanceNumber' % node_name, int(step_data.get('instance', '0')))
    cmds.setAttr('%s.instanceNumber' % node_name, lock=1)
    cmds.setAttr('%s.LevelOfDetail' % node_name, lod)
    cmds.setAttr('%s.LevelOfDetail' % node_name, lock=1)

    if step_data.get('children'):
        hide_asset_in_parent(assembly_name, step_data['children'])


def hide_asset_in_parent(assembly, children):

    parent_namespace = cmds.getAttr('%s.representationNamespace' % assembly)
    for child in children:
        node_name = '%s:%s' % (parent_namespace, child)
        cmds.assembly(node_name, a='Disabled', edit=True)



