#///////////////////////////////////////////////////////////////////////////////////////////
#//
#//  aiShadersToRedshift script 0.1 — Copyright 2017 Anders Svensson.  All rights reserved.
#//
#///////////////////////////////////////////////////////////////////////////////////////////


import maya.cmds as cmds
import math

replaceShaders = True
targetShaders = ['RedshiftMaterial', 'RedshiftMaterialBlender']

#mapping ['from', 'to']
mappingAiStandardSurface = [
            ['base', 'diffuse_weight'],
            ['baseColor', 'diffuse_color'],
            ['diffuseRoughness', 'diffuse_roughness'],
            ['specular', 'refl_weight'],
            ['specularColor', 'refl_color'],
            ['specularRoughness', 'refl_roughness'],
            ['specularIOR', 'refl_ior'],
            ['specularAnisotropy', 'refl_aniso'],
            ['specularRotation', 'refl_aniso_rotation'],
            ['metalness', 'refl_metalness'],
            ['transmission', 'refr_weight'],
            ['transmissionColor', 'refr_color'],
            ['transmissionDepth', 'refr_depth'],
            ['transmissionDispersion', 'refr_abbe'],
            ['transmissionExtraRoughness', 'refr_roughness'],
            ['transmissionScatter', 'refr_absorbtion_scale'],
            #['transmissionScatterAnisotropy', 'refr_'],
            ['coat', 'coat_weight'],
            ['coatColor', 'coat_color'],
            ['coatRoughness', 'coat_roughness'],
            ['coatIOR', 'coat_ior'],
            ['coatNormal', 'coat_bump_input'],
            ['emission', 'emission_weight'],
            ['emissionColor', 'emission_color'],
            ['opacity', 'opacity_color'],

            ['useFresnel', 'refr_use_base_IOR'],

            #['subsurface', 'transl_weight'],
            #['subsurfaceColor', 'transl_color'],
            #['subsurfaceRadius', 'ms_radius'],
            #['subsurfaceScale', 'ss_amount'],
            ['thinWalled', 'refr_thin_walled'],

            ['fogColor', 'refr_transmittance_color'],
            ['normalCamera', 'bump_input']
        ]


mappingAiMixShader = [
            ['mix', 'blendColor1'],
            ['shader1', 'baseColor'],
            ['shader2', 'layerColor1']
        ]


def setValue(attr, value):
    """Simplified set attribute function.

    @param attr: Attribute to set. Type will be queried dynamically
    @param value: Value to set to. Should be compatible with the attr type.
    """

    aType = None

    if cmds.objExists(attr):
        # temporarily unlock the attribute
        isLocked = cmds.getAttr(attr, lock=True)
        if isLocked:
            cmds.setAttr(attr, lock=False)

        # one last check to see if we can write to it
        if cmds.getAttr(attr, settable=True):
            attrType = cmds.getAttr(attr, type=True)

            #print value, type(value)

            if attrType in ['string']:
                aType = 'string'
                cmds.setAttr(attr, value, type=aType)

            elif attrType in ['long', 'short', 'float', 'byte', 'double', 'doubleAngle', 'doubleLinear', 'bool']:
                aType = None
                cmds.setAttr(attr, value)

            elif attrType in ['long2', 'short2', 'float2',  'double2', 'long3', 'short3', 'float3',  'double3']:
                if isinstance(value, float):
                    if attrType in ['long2', 'short2', 'float2',  'double2']:
                        value = [(value,value)]
                    elif attrType in ['long3', 'short3', 'float3',  'double3']:
                        value = [(value, value, value)]

                cmds.setAttr(attr, *value[0], type=attrType)


            #else:
            #    print 'cannot yet handle that data type!!'


        if isLocked:
            # restore the lock on the attr
            cmds.setAttr(attr, lock=True)

def setupConnections(inShd, fromAttr, outShd, toAttr):
    conns = cmds.listConnections( inShd + '.' + fromAttr, d=False, s=True, plugs=True )
    if conns:
        cmds.connectAttr(conns[0], outShd + '.' + toAttr, force=True)
        return True

    return False

def shaderToMtoaMaterial(inShd, nodeType, mapping):
    """
    'Converts' a shader to arnold, using a mapping table.

    @param inShd: Shader to convert
    @type inShd: String
    @param nodeType: Arnold shader type to create
    @type nodeType: String
    @param mapping: List of attributes to map from old to new
    @type mapping: List
    """

    #print 'Converting material:', inShd

    if ':' in inShd:
        rsName = inShd.rsplit(':')[-1] + '_rs'
    else:
        rsName = inShd + '_rs'

    #print 'creating '+ aiName
    rsNode = cmds.shadingNode(nodeType, name=rsName, asShader=True)
    for chan in mapping:
        fromAttr = chan[0]
        toAttr = chan[1]

        if cmds.objExists(inShd + '.' + fromAttr):
            #print '\t', fromAttr, ' -> ', toAttr

            if not setupConnections(inShd, fromAttr, rsNode, toAttr):
                # copy the values
                val = cmds.getAttr(inShd + '.' + fromAttr)

                if fromAttr == 'specularAnisotropy':
                    value = (value - 1,0) / 2.0
                elif fromAttr == 'specularAnisotropy':
                    value = (value - 1,0) / 2.0

                setValue(rsNode + '.' + toAttr, val)

    #print 'Done. New shader is ', aiNode

    return rsNode

def assignToNewShader(oldShd, newShd):
    """
    Creates a shading group for the new shader, and assigns members of the old shader to it

    @param oldShd: Old shader to upgrade
    @type oldShd: String
    @param newShd: New shader
    @type newShd: String
    """

    retVal = False

    shdGroup = cmds.listConnections(oldShd, type="shadingEngine")

    #print 'shdGroup:', shdGroup

    if shdGroup:
        cmds.connectAttr(newShd + '.outColor', shdGroup[0] + '.surfaceShader', force=True)
        cmds.delete(oldShd)

        retVal = True

    return retVal

def doMapping(inShd):
    """
    Figures out which attribute mapping to use, and does the thing.

    @param inShd: Shader name
    @type inShd: String
    """
    ret = None

    shaderType = cmds.objectType(inShd)

    if 'RedshiftMaterial' in shaderType :
        mapping = []
        for attribute_map in mappingAiStandardSurface:
            new_map = [attribute_map[1], attribute_map[0]]
            mapping.append(new_map)
        ret = shaderToMtoaMaterial(inShd, 'aiStandardSurface', mapping)

    elif 'aiMixShader' in shaderType :
        ret = shaderToMtoaMaterial(inShd, 'RedshiftMaterialBlender', mappingAiMixShader)

    if ret:
        # assign objects to the new shader
        assignToNewShader(inShd, ret)

def convertAllShaders():
    """
    Converts each (in-use) material in the scene
    """
    # better to loop over the types instead of calling
    # ls -type targetShader
    # if a shader in the list is not registered (i.e. VrayMtl)
    # everything would fail

    for shdType in targetShaders:
        print(shdType)
        shaderColl = cmds.ls(exactType=shdType)
        print(shaderColl)
        if shaderColl:
            for x in shaderColl:
                # query the objects assigned to the shader
                # only convert things with members
                shdGroup = cmds.listConnections(x, type="shadingEngine")
                setMem = cmds.sets( shdGroup, query=True )
                if setMem:
                    ret = doMapping(x)
