import os
import logging
from pprint import pprint
import maya.cmds as cmds

logger = logging.getLogger(__name__)

SHADING_INFO_ATTR = "shading_io_info"
FORMAT_VERSION = 1  # allows evolution of shading info
FORMAT_VERSION_ATTR = "{ format version }"


def save( meshListByShader, shadingFilename):
    import library.core.config_manager as config_manager

    project = os.environ['PROJECT']
    config_manager = config_manager.ConfigSolver(project=project)

    config_data = config_manager.get_config('shading_config', module='actions')
    shading_info_attribute = config_data['shading_info_attribute']
    logger.debug('Shading info attribute: %s' % shading_info_attribute)
    for shader, meshes in meshListByShader.items():

        shading_info = { config_data['format_version_attr']: config_data['format_version'] }

        for mesh in meshes:

            transform = cmds.listRelatives(mesh, parent=True)[0]
            transform_basename = transform.split(':')[-1]
            shading_info[transform_basename] = {}
            for attr in config_data['mesh_attributes']:
                shading_info[transform_basename][attr] = cmds.getAttr('%s.%s' % (transform, attr))

        if shading_info_attribute not in cmds.listAttr( shader ):
            logger.debug('Create shading info attribute: %s' % shader)
            cmds.addAttr( shader, longName = shading_info_attribute, dataType = 'string' )
        cmds.setAttr( shader + '.' + shading_info_attribute, repr( shading_info ), type = 'string' )

    #   Export shaders
    #
    cmds.select( list(meshListByShader.keys()), noExpand = True, replace = True )

    folder = os.path.dirname(shadingFilename)
    if not os.path.exists(folder):
        os.makedirs(folder)

    cmds.file(shadingFilename,
              type="mayaBinary",
              force=True,
              exportSelected=True,
              shader=True,
              channels=True,
              expressions=True,
              constraints=True,
              constructionHistory=True,
              preserveReferences=False)

    #   Clean info attributes
    #
    #

def export_shaders(output_path=''):
    ignore_sg = ['initialShadingGroup', 'initialParticleSE']
    all_shading_groups = cmds.ls(type='shadingEngine')

    shading = {}
    for sg in all_shading_groups:
        if sg in ignore_sg:
            continue
        connected = cmds.listConnections(sg, d=False, s=True)
        shading[sg] = []
        for connection in connected:
            if cmds.nodeType(connection) == 'transform':
                children = cmds.listRelatives(connection, c=True, f=True)
                if not children:
                    continue

                connection = children[0]

            print(connection)

            if cmds.nodeType(connection) != 'mesh':
                continue

            shading[sg].append(connection)

    save(shading, output_path)
    return {'shading_info': shading}

if __name__ == '__main__':
    import argparse
    logger.info('Running export shaders')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-s', '-source', '--source-file', dest='source', help='Source file path')
    parser.add_argument('-o', '-output', '--output-file', dest='output', help='Output file path')

    args = parser.parse_args()

    import maya.standalone as standalone

    standalone.initialize(name='python')

    cmds.file(args.source, o=True)

    export_shaders(args.output)

