import logging
import importlib


import usd.lib.usd_manager as usd_manager
from actions.maya.general import maya_sets_usd_collections as maya_inputs
importlib.reload(usd_manager)

logger = logging.getLogger(__name__)

IGNORE_SETS = ['defaultLightSet', 'defaultObjectSet', 
               'initialParticleSE', 'initialShadingGroup', 
               'lambert1SG', 'Model_sets']

def export_model(render_geo,
                 proxy_geo,
                 asset_name,
                 asset_type,
                 output_path,
                 model_variant,
                 model_version,
                 model_shotgrid_id,
                 project,
                 animated=False,
                 start_frame=1,
                 end_frame=24
                 ):


    import maya.cmds as cmds

    if animated:
        animated = 1
    else:
        animated = 0

    cmds.select(render_geo, r=True)
    cmds.select(proxy_geo, add=True)
    usd_render_path = render_geo.replace('|','/')
    usd_proxy_path = proxy_geo.replace('|', '/')

    options = ';exportUVs=1'
    options += ';exportSkels=none'
    options += ';exportSkin=none'
    options += ';exportBlendShapes=0'
    options += ';exportDisplayColor=0;'
    options += ';exportColorSets=1'
    options += ';exportComponentTags=1'
    options += ';defaultMeshScheme=catmullClark'
    options += f';animation={animated}'
    options += ';eulerFilter=0'
    options += ';staticSingleSample=0'
    options += f';startTime={start_frame}'
    options += f';endTime={end_frame}'
    options += ';frameStride=1'
    options += ';frameSample=0.0'
    options += ';defaultUSDFormat=usda'
    options += ';parentScope='
    options += ';shadingMode=useRegistry'
    options += ';convertMaterialsTo=[]'
    options += ';exportRelativeTextures=automatic'
    options += ';exportInstances=1'
    options += ';exportVisibility=1'
    options += ';mergeTransformAndShape=1'
    options += ';stripNamespaces=1'
    options += ';worldspace=0'

    cmds.file(output_path, options=options, typ='USD Export', es=True, pr=True)
    
    manager = usd_manager.UsdManager(project)
    manager.set_entity(asset_name, 'Asset', asset_type=asset_type)
    manager.open(output_path)

    manager.set_default_prim('/asset')
    manager.set_prim_kind('/asset', 'component')

    manager.asset_info = {'asset_name': asset_name,
                        'asset_type': asset_type,
                        'model_variant': model_variant,
                        'model_version': model_version,
                        'model_shotgrid_id': model_shotgrid_id,
                        'model_publish_path': output_path
                        }

    logger.info('render prim path: %s' % usd_render_path)
    manager.set_prim_purpose(usd_render_path, 'render')

    logger.info('proxy prim path: %s' % usd_proxy_path)
    manager.set_prim_purpose(usd_proxy_path, 'proxy')    
    
    # Function to manage the extra model options
    manage_extra_model_options(manager)

    manager.save_stage()

    logger.info(f'Usd geometry layer saved as: {output_path}')

    return {'usd': output_path}


def manage_extra_model_options(manager):
    clean_no_necessary_geom_subsets(manager)
    
    add_maya_sets_as_custom_attr(manager)
    collection_data = maya_inputs.maya_set_to_usd_collection(manager)
    check_double_side_attr(manager, collection_data)

    add_subdivision_data(manager)


def clean_no_necessary_geom_subsets(manager):
    geom_subset_prims = [i for i in manager.stage.Traverse() if i.GetTypeName() == 'GeomSubset']
    if geom_subset_prims:
        for prim in geom_subset_prims:
            if prim.GetName() in ['back', 'bottom', 'front', 'left', 'right', 'top']:
                manager.delete_prim(prim)


def add_maya_sets_as_custom_attr(manager):
    import maya.cmds as cmds
    root_prim =  manager.get_default_prim()
    custom_sets = [a for a in cmds.ls(type='objectSet') if a not in IGNORE_SETS and cmds.objectType(a) == 'objectSet']
    manager.create_attribute(root_prim, 
                            'maya_inputs',
                            attribute_type='string_array',
                            value=custom_sets)
    

def check_double_side_attr(manager, collection_data):
    if not collection_data:
        return
    elif 'transparent_meshes' in collection_data.keys():
        for maya_node in collection_data['transparent_meshes']:
            prim_path = maya_node.replace('|', '/')
            prim = manager.stage.GetPrimAtPath(prim_path)
            if not prim.IsValid():
                continue
            ds_attr = prim.GetAttribute('doubleSided').Set(False)


def add_subdivision_data(manager):
    import maya.cmds as cmds

    subdiv_type_rel = {0: 'none', 1: 'catclark', 2: 'catclark'}
    stored_meshes = dict()

    for mesh in cmds.ls(type='mesh', l=True):
        sub_type = subdiv_type_rel[cmds.getAttr(f"{mesh}.aiSubdivType")]
        if sub_type != 'none':
            subdiv_iterations = 1
        else:
            subdiv_iterations = 0
        mesh_transform_name = "/".join(mesh.split('|')[:-1])
        stored_meshes[mesh_transform_name] = {'subdiv_type': sub_type, 
                                              'subdiv_iterations': subdiv_iterations}

    for prim in manager.stage.Traverse():
        if prim.GetTypeName() != 'Mesh':
            continue
        prim_path = str(prim.GetPath())
        manager.create_attribute(prim, 
                                 'primvars:arnold:subdiv_type',
                                 attribute_type='token',
                                 value=stored_meshes[prim_path]['subdiv_type'])
        manager.create_attribute(prim, 
                                 'primvars:arnold:subdiv_iterations',
                                 attribute_type='uchar',
                                 value=stored_meshes[prim_path]['subdiv_iterations'])


if __name__ == '__main__' and False:
    import argparse

    print('Running export usd')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-o', '-output', '--output-file', dest='output', help='Output file path')
    parser.add_argument('-g', '-geo-path', '--geo-path', dest='geo_path', help='render geometry')
    parser.add_argument('-p', '-proxy-path', '--proxy-path', dest='proxy_path', help='proxy geometry')

    args = parser.parse_args()

    create_usd_model(output_path=args.output, geo_path=args.geo_path, proxy_path=args.proxy_path)


if __name__ == '__main__' :

    output_path = r'V:\SGD\publish\model\armChairGamer\Master\armChairGamer_Master_0f4830702a4860c9\v044\usd/armChairGamer_geometry.usda'
    geo_path = r'V:\SGD\publish\model\armChairGamer\Master\armChairGamer_Master_0f4830702a4860c9\v044\abc/armChairGamer_high.abc'
    proxy_path = r'V:\SGD\publish\model\armChairGamer\Master\armChairGamer_Master_0f4830702a4860c9\v044\abc/armChairGamer_low.abc'
    create_usd_model(output_path=output_path,
                         geo_path=geo_path,
                         proxy_path=proxy_path,
                         asset_name='armChairGamer',
                         variant='Master',
                         version=44)
