import importlib
import os

import movie_transfer.bin.create_quicktime as create_quicktime
import maya.cmds as cmds
import maya.mel as mel
import mayaUsd

importlib.reload(create_quicktime)

def viewport (sw, model_panel='modelPanel4') :
    """
      viewport UI commands
    """
    if not cmds.modelPanel(model_panel, q=True, ex=True):
        cmds.modelPanel(model_panel)

    if "modelPanel" not in model_panel:
        raise RuntimeError ("No active model panel found..")
    if sw == 'ON':
        mel.eval("paneLayout -e -manage true $gMainPane")
        cmds.select(clear=True)
    if sw == 'OFF' :
        mel.eval("paneLayout -e -manage false $gMainPane")
    if sw == 'basicGL' :
        cmds.modelEditor(model_panel, e=True, rnm='base_OpenGL_Renderer')
    if sw == 'vp2' :
        cmds.ActivateViewport20()
    if sw == 'defaultlight_textured':
        cmds.modelEditor(model_panel, e=1, allObjects=0)
        cmds.modelEditor(model_panel, e=1, polymeshes=1)
        cmds.modelEditor(model_panel, e=1, strokes=1)
        cmds.modelEditor(model_panel, e=1, xray=0)
        cmds.modelEditor(model_panel, e=1, displayTextures=1)
        cmds.modelEditor(model_panel, e=1, displayLights='default')
        cmds.modelEditor(model_panel, e=1, dimensions=0)
        cmds.modelEditor(model_panel, e=1, pluginShapes=True)
        cmds.modelEditor(model_panel, e=1, displayAppearance='smoothShaded')

    if sw == 'boundingBox':
        cmds.modelEditor(model_panel, e=1, displayAppearance='boundingBox')

def find_camera(shot_name):
    camera_names = ['Camera_NS:*',
                    '*SHOT:cameraShap*',
                    '*CAM_NS:cameraShap*',
                    '*SHOT_NS:cameraShap*',
                    '%s*' % shot_name,
                    'perspShape']
    print(camera_names)
    for camera_name in camera_names:
        cameras = cmds.ls(camera_name, type='camera')
        if cameras:
            print(cameras)
            return cameras[0]

    for usd_shape in cmds.ls(type="mayaUsdProxyShape", long=True):
        camera_path = '/World/Camera/%s/%sShape' % (shot_name, shot_name)
        camera = '%s,%s' % (usd_shape, camera_path)
        return camera

    return 'Camera_SHOT_NS:cameraShape1'

def setMayaTimeline( cTime = 0, min = 0, max = 0) :
    if isinstance( min , int ) and isinstance( max , int ) :
        cmds.playbackOptions ( edit=True, min = min, max = max )
        cmds.playbackOptions ( edit=True, ast = min, aet = max )
        try    : cmds.currentTime(      cTime, update=True)
        except : pass

def launch_playblast(shot_name='',
                     output_path='',
                     images_path='',
                     start_frame=101,
                     end_frame=110,
                     isLayout=False,
                     ):
    # Get the Resolution from Maya scene
    #
    widthMov = cmds.getAttr("defaultResolution.width")
    heightMov = cmds.getAttr("defaultResolution.height")

    # Prepare the playblast and launch it
    aPlayBackSliderPython = mel.eval('$tmpVar=$gPlayBackSlider')
    #

    setMayaTimeline(start_frame, start_frame, end_frame)

    cameraShape = find_camera(shot_name)

    print('CAMERA SHAPE', cameraShape)

    # Enable the shot sound in timeline

    sound = cmds.timeControl(aPlayBackSliderPython, query=True, sound=True)
    cmds.lookThru(cameraShape)

    view = cmds.playblast(activeEditor=True)

    viewport("defaultlight_textured", model_panel=view)

    cmds.setAttr('%s.displayResolution' % cameraShape, 0)
    cmds.setAttr('%s.displayFilmGate' % cameraShape, 0)

    cmds.setAttr('%s.overscan' % cameraShape, 1.0)

    folder = os.path.dirname(output_path)
    temp_file = os.path.join(folder, 'tmp')
    images_folder = os.path.dirname(images_path)

    if not os.path.exists(images_folder):
        os.makedirs(images_folder)

    cmds.playblast(s=sound,
                   f=images_path,
                   cc=True,
                   w=widthMov, h=heightMov,
                   v=False,
                   p=100,
                   fmt='image',
                   st=int(start_frame),
                   et=int(end_frame),
                   fo=True,
                   os=True,
                   compression="png",
                   sequenceTime=isLayout)

    output_folder = os.path.dirname(output_path)
    print(temp_file)
    print(output_folder)

    create_quicktime.create_quicktime(folder,
                                      output_path,
                                      top_center_text=shot_name,
                                      top_left_text=None,
                                      bottom_left_text=None,
                                      first_frame=start_frame,
                                      last_frame=end_frame,
                                      delete_sources=True,
                                      aov_list=['temp']
                                      )

    return {'playblast': output_path}