import os
from pprint import pprint
import subprocess

import shotgrid_lib.database as sg_database
import publisher.lib.publish_cmd as publisher_cmd

cut_in_keys = ['Timecode In', 'TCIN']
cut_out_keys = ['Timecode Out', 'TCOUT']


def timecode_to_frames(timecode, framerate=25):
    split_tc = timecode.split(':')
    if len(split_tc) == 3:
        split_tc.insert(0, '00')
    return sum(f * int(t) for f, t in zip((3600*framerate, 60*framerate, framerate, 1), split_tc))

def get_frames(edl_data):
    cut_in_tc_key = None
    cut_out_tc_key = None
    first_item = list(edl_data.values())[0]
    for key in cut_in_keys:
        if key in first_item:
            cut_in_tc_key = key

    for key in cut_out_keys:
        if key in first_item:
            cut_out_tc_key = key

    print('Cut in key:', cut_in_tc_key)
    print('Cut out key:', cut_out_tc_key)
    for item in edl_data.values():
        cut_in_tc = item[cut_in_tc_key]
        cut_out_tc = item[cut_out_tc_key]
        first_frame = timecode_to_frames(cut_in_tc)
        last_frame = timecode_to_frames(cut_out_tc) - 1
        frames = last_frame - first_frame + 1

        item['cut_in'] = first_frame
        item['cut_out'] = last_frame + 1
        item['first_frame'] = 1001
        item['last_frame'] = 1000 + frames
        item['duration'] = frames


def read_edl(path):
    import csv
    with open(path, newline='') as csvfile:
        edl_data = csv.reader(csvfile, delimiter=';', quotechar='|')

        keys = None
        all_data = {}
        for row in edl_data:
            if keys is None:
                keys = row
            else:
                if not row[0]:
                    continue

                all_data[row[0]] = {}
                for key, value in zip(keys, row):
                    if not key:
                        continue
                    if value.isdigit():
                        value = int(value)
                    all_data[row[0]][key] = value


        #get_frames(all_data)
        return all_data

def get_timecode(timecode, fps=25.0):

    minutes, seconds, frame = timecode.rsplit(':')
    milisecond = float(frame) / fps
    seconds = int(seconds) + milisecond
    print(timecode, seconds)
    if seconds < 10.0:
        new_tc = '00:%s:0%.3f' % (minutes, seconds)
    else:
        new_tc = '00:%s:%2.3f' % (minutes, seconds)
    return new_tc


def run_subprocess(cmd):
    sp = subprocess.Popen(cmd,
                          stderr=subprocess.PIPE,
                          stdout=subprocess.PIPE,
                          env=os.environ.copy())  # FFmpeg uses stderr in place of stdout
    out, err = sp.communicate()
    if out:
        print("standard output of subprocess b:")
        for line in str(out.decode("utf-8")).split('\n'):
            print(line)
    if err:
        print("standard error of subprocess b:")
        print(err)
        for line in str(err.decode("utf-8")).split('\n'):
            print(line)


def cut_audio(video_path, edl_data, shot_name, output_path):
    dirname = os.path.dirname(output_path)

    if not os.path.exists(dirname):
        os.makedirs(dirname)

    for qt_basename, qt_data in edl_data.items():
        name = qt_basename.split('.')[0]
        if shot_name and shot_name != name:
            continue
        extract_audio = 'C:/ProgramData/chocolatey/bin/ffmpeg.exe -i %s -vn  %s' %(video_path, output_path)
        run_subprocess(extract_audio)

    return ()

def cut_video(video_path, edl_data, shot_name, output_path):
    dirname = os.path.dirname(output_path)

    if not os.path.exists(dirname):
        os.makedirs(dirname)


    for qt_basename, qt_data in edl_data.items():
        name = qt_basename.split('.')[0]
        if shot_name and shot_name != name:
            continue

        cmd_line = 'C:/ProgramData/chocolatey/bin/ffmpeg.exe -i %s -ss %s -to %s -c:v copy -c:a copy %s'
        timecode_in = get_timecode(qt_data['TCIN'])
        timecode_out = get_timecode(qt_data['TCOUT'])


        ffMpgCmd = cmd_line % (video_path,
                               timecode_in,
                               timecode_out,
                               output_path)
        run_subprocess(ffMpgCmd)


def extract_video(shot_name, edl_path, video_path, output_path):
    edl_data = read_edl(edl_path)
    cut_video(video_path, edl_data, shot_name, output_path)

    first_frame = 1001
    duration = edl_data['%s.mov' % shot_name]['DURATION']
    last_frame = first_frame + duration -1

    return {'first_frame': first_frame,
            'last_frame': last_frame}

def extract_audio(shot_name, edl_path, video_path, output_path):
    edl_data = read_edl(edl_path)
    cut_audio(video_path, edl_data, shot_name, output_path)

    first_frame = 1001
    duration = edl_data['%s.mov' % shot_name]['DURATION']
    last_frame = first_frame + duration -1
    return {'first_frame': first_frame,
            'last_frame': last_frame}


def create_sequence(database, sequence_name, episode_view):
    publish_record = sg_database.Record('Sequence')
    publish_record.code = sequence_name
    publish_record.episode = episode_view
    publish_record.project = episode_view.project
    database.append(publish_record)

def create_shot(database, shot_name, sequence_view, shot_template):
    publish_record = sg_database.Record('Shot')
    publish_record.code = shot_name
    publish_record.sg_sequence = sequence_view
    publish_record.project = sequence_view.project
    publish_record.task_template = shot_template
    database.append(publish_record)


def create_shots(project, episode, edl_tree, edl_path, movie_path, publish_id):

    database = sg_database.DataBase()
    database.fill(project, precatch=False)

    database.query_sg_database('Episode')
    database.query_sg_database('Sequence')
    database.query_sg_database('Shot')
    database.query_sg_database('TaskTemplate')
    shot_template = database['TaskTemplate']['Animation - Shot']
    episode_view = database['Episode'][episode]
    print(episode_view)
    for sequence_name, sequence_shots in edl_tree.items():
        sequence_view = database['Sequence'][sequence_name]
        if sequence_view.empty:
            create_sequence(database, sequence_name, episode_view)
            sequence_view = database['Sequence'][sequence_name]

        for shot_name, shot_data in sequence_shots.items():
            shot_view = database['Shot'][shot_name]
            if shot_view.empty:
                create_shot(database, shot_name, sequence_view, shot_template)
                shot_view = database['Shot'][shot_name]

            shot_view.sg_cut_in = shot_data['first_frame']
            shot_view.sg_cut_out = shot_data['last_frame']
            shot_view.sg_cut_duration = shot_data['DURATION']
            shot_view.update_shotgun()
            shot_view.precache_dependencies(fields=['tasks'])

            parameters = {'entity_name': shot_name,
                          'entity_type': 'Shot',
                          'comment': 'launched from inject edl',
                          'task_code': 'Animatic',
                          'source_path': edl_path,
                          'video_path': movie_path,
                          'project': project,


                          }
            pprint(parameters)

            publisher_cmd.PublishCmd('Animatic', parameters, project)

def create_shotgrid_shots(project, episode, edl_data, edl_path, movie_path, publish_id):

    episode_tree = {}
    for shot_name, shot_data in edl_data.items():
        shot_name = shot_name.split('.')[0]
        bits = shot_name.split('_')
        sequence_number = bits[2]
        shot_number = bits[3]
        sequence_name = '%s_%s' % (episode, sequence_number)
        shot_name = '%s_%s' % (sequence_name, shot_number)

        shotgrid_shot_data = shot_data.copy()

        episode_tree[sequence_name] = episode_tree.get(sequence_name, {})

        shotgrid_shot_data['first_frame'] = 1001
        shotgrid_shot_data['last_frame'] = 1000 + shot_data['DURATION']
        shotgrid_shot_data['shot'] = shot_name
        shotgrid_shot_data['sequence'] = sequence_name
        shotgrid_shot_data['episode'] = episode
        episode_tree[sequence_name][shot_name] = shotgrid_shot_data

    create_shots(project, episode, episode_tree, edl_path, movie_path, publish_id)


def inject_edl(project, episode, edl_path, movie_path, shotgrid_id):
    edl_data = read_edl(edl_path)
    create_shotgrid_shots(project, episode, edl_data, edl_path, movie_path, shotgrid_id)



if __name__ == '__main__':
    path = 'C:/projects/TPT/documents/s01_ep00_edl.csv'
    source_file = 'C:/projects/TPT/documents/SY_Trailer_Animatic_Son_019.mov'
    edl_data = read_edl(path)

    episode = 's00_ep02'
    project = 'TPT'
    pprint(edl_data)
    #create_shotgrid_shots(project, episode, edl_data, path, source_file)
    #cut_video(source_file, edl_data)