import os

import maya.cmds as cmds
import usd.lib.usd_manager as usd_manager


def create_baked_camera(camera_name, shot_name, start_frame, end_frame):
    camera_children = cmds.listRelatives(camera_name, c=True)
    if not camera_children :
        return

    original_camera_shape = None
    for cam in camera_children:
        if cmds.nodeType(cam) == 'camera':
            original_camera_shape = cam
    if original_camera_shape is None:
        return
    print(original_camera_shape)

    world_node = cmds.createNode('transform', name='World_tmp')
    cameras_node = cmds.createNode('transform', name='Camera', parent=world_node)
    shot_cameras_node = cmds.createNode('transform', name=shot_name, parent=cameras_node)

    camera_node = cmds.createNode('camera', name='%sShape' % shot_name, parent=shot_cameras_node)

    cmds.parentConstraint(camera_name, shot_cameras_node, mo=False)
    connect_attributes = ['focalLength',
                          'cameraScale',
                          'farClipPlane',
                          'nearClipPlane',
                          'horizontalFilmAperture',
                          'verticalFilmAperture',
                          'depthOfField',
                          'focusDistance',
                          'fStop',
                          'focusRegionScale',
                          ]

    for attr in connect_attributes:
        cmds.connectAttr('%s.%s' % (original_camera_shape, attr), '%s.%s' % (camera_node, attr),f=True)


    bake_nodes = [shot_cameras_node, camera_node]
    at = ['tx', 'ty', 'tz',
          'rx', 'ry', 'rz',
          'sx', 'sy', 'sz']
    at = at + connect_attributes
    #cmds.bakeSimulation(bake_nodes, t=(start_frame, end_frame), at=at, hi="below")
    cmds.bakeResults(bake_nodes, t=(start_frame, end_frame), simulation=True)

    return camera_node


def export_camera(output_path,
                  shot_name,
                  camera_node='',
                  start_frame=101,
                  end_frame=110,
                  preroll=10,
                  post_roll=10,
                  project=''
                  ):

    if not camera_node:
        all_cameras = cmds.ls(type='camera', long=True)
        for camera_name in all_cameras:
            parent_name, short_name = camera_name.rsplit('|', 1)
            if short_name.find('_SHOT_') > -1:
                camera_node = parent_name
    print('camera_node', camera_node)
    start_frame = start_frame - preroll
    end_frame = end_frame + post_roll

    baked_camera = create_baked_camera(camera_node, shot_name, start_frame, end_frame)

    cmds.select(baked_camera, r=True)

    options = {'exportUVs': '1',
               'exportSkels': 'none',
               'exportSkin': 'none',
               'exportBlendShapes': '0',
               'exportDisplayColor': '0',
               'exportColorSets': '1',
               'exportComponentTags': '1',
               'defaultMeshScheme': 'catmullClark',
               'animation': '1',
               'eulerFilter': '0',
               'staticSingleSample': '0',
               'startTime': start_frame,
               'endTime': end_frame,
               'frameStride': '1',
               'frameSample': '1.0',
               'defaultUSDFormat': 'usda',
               'parentScope': '',
               'shadingMode': 'useRegistry',
               'convertMaterialsTo': '[]',
               'exportRelativeTextures': 'automatic',
               'exportInstances': '1',
               'exportVisibility': '1',
               'mergeTransformAndShape': '1',
               'stripNamespaces': '1',
               'worldspace': '0'}

    folder = os.path.dirname(output_path)
    if not os.path.exists(folder):
        os.makedirs(folder)

    options_str = ';'
    for key, value in options.items():
        options_str += '%s=%s;' % (key, str(value))

    cmds.file(output_path, options=options_str, typ='USD Export', es=True, pr=True, f=True)
    manager = usd_manager.UsdManager(project)
    manager.set_entity(shot_name, 'Shot')
    manager.open(output_path)
    manager.repath_prim('/World_tmp', '/World')

    manager.stage.SetMetadata('timeCodesPerSecond', 24)

    manager.save_stage()


    cmds.delete('World_tmp')
    return{'usd': output_path}


if __name__ == '__main__':
    import argparse

    print('Running export camera usd')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-so', '-source', '--source-file', dest='source', help='Source file path')
    parser.add_argument('-p', '-project', '--project', dest='project', help='Project code')
    parser.add_argument('-o', '-output', '--output-path', dest='output_path', help='Output file folder')
    parser.add_argument('-sh', '-shot', '--shot-name', dest='shot_name', help='Output file folder')
    parser.add_argument('-sf', '-start', '--start-frame', dest='start_frame', help='Output file folder')
    parser.add_argument('-ef', '-end', '--end-frame', dest='end_frame', help='Output file folder')

    args = parser.parse_args()

    import maya.standalone as standalone

    standalone.initialize(name='python')

    cmds.file(args.source, o=True)
    output_paths = export_camera(args.output,
                                 args.shot_name,
                                 project=args.project,
                                 start_frame=args.start_frame,
                                 end_frame=args.end_frame,
                                 )

