import importlib
import os.path
from pprint import pprint

import maya.cmds as cmds

ASSET_NODES={'Others': {'suffix': 'SHOT', 'group': 'Shot_elements'},
             'Main Characters': {'suffix': 'CHAR', 'group':'Characters'},
             'Secondary Characters': {'suffix': 'CHAR', 'group': 'Characters'},
             'Setprops': {'suffix': 'PROP', 'group': 'Props'},
             'Props': {'suffix': 'PROP', 'group': 'Props'},
             'Sets': {'suffix': 'SET', 'group':'Sets'},
        }

IGNORE_ASSETS = ['Audio']
IGNORE_ASSET_TYPES = ['Sets']

def get_atom_namespace(path, scene_namespace):

    atom_file = open(path, 'r')

    for line in atom_file.readlines():
        if line.find(':') > -1:
            bits = line.split(' ')
            for bit in bits:
                if bit.find(':') > -1:
                    new_namespace, node_name = bit.split(':', 1)
                    scene_node = '%s:%s' % (scene_namespace, node_name)
                    if cmds.objExists(scene_node):
                        atom_file.close()
                        return new_namespace

    return None

def import_atom_file(atom_path, atom_namespace, scene_namespace):

    opti0ns_dict = {'targetTime':3,
                   'option':'scaleReplace',
                   'match': 'string',
                   'selected': 'selectedOnly',
                   'search': atom_namespace,
                   'replace': scene_namespace,
                   }

    options = []
    for key, value in opti0ns_dict.items():
        options.append('%s=%s' % (key, value))

    options_string = ';'.join(options)

    cmds.file(atom_path, i=True, type='atomImport', ra=True, options=options_string)

def get_scene_namespace():

    selected_nodes = cmds.ls(sl=True)
    for node in selected_nodes:
        if node.find(':') == -1:
            continue
        namespace = node.split(':')[0]
        return namespace

def appy_atom_file(atom_path, scene_namespace=None):

    if scene_namespace is None:
        scene_namespace = get_scene_namespace()

    atom_namespace = get_atom_namespace(atom_path, scene_namespace)
    print('apply atom file: %s' % atom_path)
    print('scene_namespace: %s' % scene_namespace)
    print('atom_namespace: %s' % atom_namespace)

    import_atom_file(atom_path, atom_namespace, scene_namespace)



def apply_layout(step_data=None, project='', current_step='', load_camera=True):

    print('== apply layout ==')
    if load_camera:
        import actions.maya.animation.load_camera as load_camera
        importlib.reload(load_camera)
        load_camera.load_camera_cache(step_data=step_data, project=project, current_step=current_step)

    if not cmds.pluginInfo('atomImportExport', q=True, l=True):
        cmds.loadPlugin('atomImportExport')

    for alias, asset_breakdown in step_data['breakdown'].items():
        print('-' * 100)
        asset_name = asset_breakdown['asset_name']
        if asset_name in IGNORE_ASSETS:
            continue
        print('%s_animation' % asset_name)

        asset_type = asset_breakdown.get('asset_type')
        if asset_type in IGNORE_ASSET_TYPES:
            continue

        instance = asset_breakdown.get('instance_number', 1)
        default = {'suffix': 'DNF', 'group': 'others'}
        sufix = ASSET_NODES.get(asset_type, default)['suffix']

        if alias:
            root_node = '%s_%s' % (alias, sufix)
        else:
            root_node = '%s_%03d_%s' % (asset_name, instance, sufix)

        print('root node')
        if not cmds.objExists(root_node):
            continue
        if asset_name == 'Camera':
            file_tab = 'Camera_001_animation'
        else:

            file_tab = '%s_%03d_animation' % (asset_name, instance)

        namespace = cmds.getAttr('%s.representationNamespace' % root_node)

        print('namespace', namespace)
        print('file_tab', file_tab)

        animation_path = step_data['files'].get(file_tab)
        print(animation_path)
        if not animation_path:
            continue

        full_path = '%s/%s' % (step_data['published_folder'], animation_path)
        print(full_path)
        if not os.path.exists(full_path):
            continue

        print('root node', root_node)
        if not cmds.objExists(root_node):
            continue

        controls_set = '%s:Rig_Controls' % namespace
        print(controls_set)
        if not cmds.objExists(controls_set):
            continue

        cmds.select(controls_set, r=True)

        appy_atom_file(full_path, scene_namespace=namespace)

