import os

import maya.cmds as cmds
import shotgrid_lib.database as database

ASSET_TYPES = {'CHAR': ['Main Characters', 'Secondary Characters', 'Tertiaty Characters',
                        'Main Creatures', 'Secondary Creatures', 'Tertiaty Creatures'
                        ],
               'PROP': ['Props', 'Setprops'],
               'CAM': ['Others']}

def export_animation(publish_folder ='', relative_folder='', rig_controls='Rig_Controls', valid_types=['PROP', 'CHAR']):
    cmds.loadPlugin("atomImportExport", quiet=True)
    if isinstance(valid_types, str):
        valid_types = valid_types.split(' ')

    valid_asset_types = []
    for valid_type in valid_types:
        valid_asset_types += ASSET_TYPES[valid_type]

    dirname = '%s/%s' % (publish_folder, relative_folder)
    if not os.path.exists(dirname):
        os.makedirs(dirname)

    all_assemblies = cmds.ls(type='RigAssembly')
    output = {'output_files':{}, 'metadata': {'assets': {}}}

    for assembly_node in all_assemblies:
        asset_type = cmds.getAttr('%s.assetType' % assembly_node)

        if asset_type not in valid_asset_types:
            continue

        namespace = cmds.getAttr('%s.representationNamespace' % assembly_node)
        asset_name = cmds.getAttr('%s.assetName' % assembly_node)
        instance_number = cmds.getAttr('%s.instanceNumber' % assembly_node)

        file_basename = '%s_%03d' % (asset_name, instance_number)
        set_name = '%s:%s' % (namespace, rig_controls)
        cmds.select(set_name, r=True)

        relative_path = '%s/%s.atom' % (relative_folder, file_basename)
        path = '%s/%s.atom' % (dirname, file_basename)

        output['output_files']['%s_animation' % file_basename] = relative_path

        output['metadata']['assets'][asset_name] = {instance_number: assembly_node}

        options = "precision=8;statics=1;baked=0;sdk=0;constraint=0;animLayers=0;selected=selectedOnly"
        print(path)
        cmds.file(path, exportSelected=True, force=True, options=options, type="atomExport")

    return output

def store_output_files(project, shotgrid_id, output_files):
    sg_database = database.DataBase()
    sg_database.fill(project, precatch=False)

    filters = [['id', 'is', int(shotgrid_id)]]
    sg_database.query_sg_database('CustomEntity09', filters=filters)
    this_record = sg_database['CustomEntity09'].find_with_filters(id=int(shotgrid_id), single_item=True)
    previous_files = this_record.sg_files

    if not previous_files:
        previous_files = {}

    for tag, path in output_files['output_files'].items():
        previous_files[tag] = path
    this_record.sg_files = str(previous_files)
    this_record.update_shotgun()


if __name__ == '__main__':
    import argparse

    print('Running export alembic')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-s', '-source', '--source-file', dest='source', help='Source file path')
    parser.add_argument('-pf', '-folder', '--publish-folder', dest='publish_folder', help='Output file folder')

    parser.add_argument('-rf', '-relative', '--relative-folder', dest='relative_folder', help='relative folder to store caches')

    parser.add_argument('-rc', '-controls', '--rig-controls', dest='rig_controls', default='Rig_Controls', help='relative folder to store caches')

    parser.add_argument('-t', '-types', '--valid-types', dest='valid_types', default=['PROP', 'CHAR'], nargs='+', help='List of valid types to cache')

    parser.add_argument('-p', '-project', '--project', dest='project', help='Project code')
    parser.add_argument('-i', '-id', '--shotgrid-id', dest='shotgrid_id', help='Current publish shotgrid ID', type=int)

    args = parser.parse_args()

    import maya.standalone as standalone

    standalone.initialize(name='python')

    valid_types = args.valid_types[0].split(' ')
    cmds.file(args.source, o=True)
    output_paths = export_animation(publish_folder=args.publish_folder,
                                    relative_folder=args.relative_folder,
                                    rig_controls=args.rig_controls,
                                    valid_types=valid_types
                                    )

    store_output_files(args.project, args.shotgrid_id, output_paths)